CREATE OR REPLACE FUNCTION code.move_messages_to_tab(
    i_uid   code.uid,
    i_mids  code.mids,
    i_dst_tab mail.tab_types,
    i_request_info code.request_info DEFAULT NULL
) RETURNS code.operation_result AS $$
DECLARE
    current_revision     constant bigint := code.acquire_current_revision(i_uid);
    moved                impl.moved_message[];
BEGIN
    /*
      1. update mail.box and define what really @moved
      2. update old and new tabs
      3. update threads revision
      4. revision increment
    */

    IF i_dst_tab IS NOT NULL AND NOT EXISTS (
        SELECT 1
          FROM mail.tabs t
         WHERE t.uid = i_uid
           AND t.tab = i_dst_tab
    ) THEN
        RAISE EXCEPTION 'Can''t move to tab % for uid %, it does not exist', i_dst_tab, i_uid;
    END IF;

    WITH with_moved AS (
        UPDATE mail.box m
           SET revision = current_revision,
               tab = i_dst_tab,
               newest_tit = impl.default_newest_tit(i_dst_tab)
          FROM (
            SELECT mid,
                   fid AS src_fid,
                   recent AS old_recent,
                   tid AS src_tid,
                   chain AS src_chain,
                   imap_id AS src_imap_id,
                   newest_tif AS old_newest_tif,
                   tab as src_tab,
                   newest_tit AS old_newest_tit
              FROM mail.box
             WHERE uid = i_uid
               AND tab IS DISTINCT FROM i_dst_tab
               AND mid = ANY(i_mids)
          ) ml
         WHERE m.uid = i_uid
           AND m.mid = ml.mid
        RETURNING m.mid, ml.src_fid,
                  m.tid, ml.src_tid,
                  m.imap_id, ml.src_imap_id,
                  ml.src_chain,
                  m.seen, ml.old_recent, m.deleted,
                  m.lids, m.received_date, old_newest_tif,
                  ml.src_tab, old_newest_tit)
    SELECT array_agg((
        um.mid, um.src_fid, um.src_fid,
        um.tid, um.src_tid,
        um.imap_id, um.src_imap_id,
        um.src_chain,
        um.seen, um.old_recent, deleted,
        um.lids,
        mm.size,
        um.received_date,
        old_newest_tif,
        impl.get_attaches_info(mm.attaches),
        hdr_message_id,
        mm.attributes,
        i_dst_tab, um.src_tab, old_newest_tit
        )::impl.moved_message)
      INTO moved
      FROM with_moved um
      JOIN mail.messages mm
        ON (um.mid = mm.mid and mm.uid = i_uid);

    IF moved IS NULL OR cardinality(moved) = 0 THEN
        RETURN code.unchanged_result();
    END IF;


    PERFORM impl.update_src_folders(
        i_uid,
        current_revision,
        moved
    );

    PERFORM impl.update_src_tabs(
        i_uid,
        current_revision,
        moved
    );

    PERFORM impl.update_threads(
        i_uid,
        current_revision,
        moved
    );

    PERFORM impl.update_dst_tab(
        i_uid,
        i_dst_tab,
        current_revision,
        moved
    );

    PERFORM impl.log_change(
        i_uid          => i_uid,
        i_request_info => i_request_info,
        i_revision     => current_revision,
        i_type         => 'move-to-tab',
        i_changed      => ARRAY(
          SELECT (mid, dst_tid, dst_fid, src_fid,
                  seen, true, deleted,
                  CASE WHEN dst_tid IS NULL
                       THEN ARRAY[]::integer[]
                       ELSE lids END,
                  i_dst_tab, src_tab
                 )::impl.changed_message_with_src_fid_tab
           FROM unnest(moved))::impl.changed_with_fids,
        i_args => jsonb_build_object('tab', i_dst_tab)
    );

    PERFORM code.increment_revision(i_uid);

    RETURN (current_revision, ARRAY(SELECT mid FROM unnest(moved)))::code.operation_result;
END;
$$ LANGUAGE plpgsql;
