CREATE OR REPLACE FUNCTION code.quick_save_message(
    i_uid        code.uid,
    i_mid        bigint,
    i_coords     code.quick_save_coordinates,
    i_headers    code.quick_save_headers,
    i_recipients code.store_recipient[],
    i_attaches   code.store_attach[],
    i_request_info code.request_info DEFAULT NULL,
    i_mime       code.store_mime_part[] DEFAULT NULL
) RETURNS SETOF code.quick_save_result AS $$
DECLARE

    i_tid bigint;
    i_lids integer[];
    i_fid integer;

    i_old_size integer := 0;
    i_old_attaches mail.attach[];

    i_attach_size integer := 0;
    i_old_attach_size integer := 0;
    i_attach_count integer := 0;
    i_old_attach_count integer := 0;
    attach mail.attach;
    ic_attach mail.attach;
    ic_attaches mail.attach[];
    ic_mime mail.mime_part[] := i_mime;

    current_revision bigint := code.acquire_current_revision(i_uid);

    ic_recipients mail.recipient[] := i_recipients;

    i_msg_accounted_by_labels bool;
    i_newest_tif bool := true;
    prev_newest_mid bigint;
    update_prev_newest bool;
    i_pop3initialized bool := false;
BEGIN

-- Get the old envelope information, if no envelope is found then just return
-- the false flag and current revision because probably message has been
-- deleted right before the function call
    SELECT me.size, me.attaches, m.fid, m.tid, m.lids
      INTO i_old_size, i_old_attaches, i_fid, i_tid, i_lids
      FROM mail.box m
      JOIN mail.messages me
        ON m.mid = me.mid
       AND me.uid = m.uid
     WHERE m.uid = i_uid
       AND m.mid = i_mid;

    IF NOT found THEN
      RETURN NEXT (false, code.unchanged_revision())::code.quick_save_result;
      RETURN;
    END IF;

-- Get attachments information for a modified envelope
    FOREACH attach IN ARRAY coalesce(i_attaches, ARRAY[]::code.store_attach[]) LOOP
        i_attach_size := i_attach_size + attach.size;
        i_attach_count := i_attach_count + 1;
        ic_attach := attach;
        ic_attaches := ic_attaches || ic_attach;
    END LOOP;

-- Get old attachments information for an existent envelope
    FOREACH attach IN ARRAY coalesce(i_old_attaches, ARRAY[]::mail.attach[]) LOOP
        i_old_attach_size := i_old_attach_size + attach.size;
        i_old_attach_count := i_old_attach_count + 1;
    END LOOP;

-- Get the new revision before make the modifications
    current_revision = code.increment_revision(i_uid);

-- Update part of messages data
    UPDATE mail.messages
       SET st_id = i_coords.st_id,
           size = i_coords.size,
           attaches = ic_attaches,
           mime = ic_mime,
           attributes = i_coords.attributes,
           firstline = i_headers.firstline,
           hdr_date = i_headers.hdr_date,
           hdr_message_id = i_headers.hdr_message_id,
           recipients = ic_recipients,
           pop_uidl = i_coords.pop_uidl
    WHERE uid = i_uid
      AND mid = i_mid;

-- Update folder info
    UPDATE mail.folders
       SET attach_count = attach_count + i_attach_count - i_old_attach_count,
           attach_size = attach_size + i_attach_size - i_old_attach_size,
           message_size = message_size + i_coords.size - i_old_size,
           revision = current_revision
     WHERE uid = i_uid
       AND fid = i_fid
    RETURNING code.is_threaded_folder(type), (pop3state).initialized
      INTO i_msg_accounted_by_labels, i_pop3initialized;

    IF NOT found THEN
        RAISE EXCEPTION 'Nonexistent folder fid: % uid: %', i_fid, i_uid
              USING TABLE = 'mail.folders';
    END IF;

-- Updating thread if the message is in a thread
    IF i_tid IS NOT NULL THEN
        UPDATE mail.threads
           SET revision = current_revision,
               newest_date = greatest(i_coords.received_date, newest_date),
               newest_mid = CASE WHEN newest_date > i_coords.received_date
                                 THEN newest_mid
                                 ELSE i_mid END,
               attach_count = attach_count + i_attach_count - i_old_attach_count,
               attach_size = attach_size + i_attach_size - i_old_attach_size
         WHERE uid = i_uid
           AND tid = i_tid;

        IF NOT found THEN
            RAISE EXCEPTION 'Got itself in trouble while updating threads'
                USING DETAIL = 'thread_exsits is true but no threads have been updated';
        END IF;

-- Updating newest in thread
        SELECT mid, received_date < i_coords.received_date
          INTO prev_newest_mid, update_prev_newest
          FROM mail.box
         WHERE uid = i_uid
           AND fid = i_fid
           AND tid = i_tid
           AND NOT mid = i_mid
           AND newest_tif;

        IF found THEN
            i_newest_tif := update_prev_newest;
            IF update_prev_newest THEN
                UPDATE mail.box
                   SET newest_tif = false
                 WHERE uid = i_uid
                   AND mid = prev_newest_mid;
            END IF;
        END IF;

    END IF;

-- Updating mailbox
    UPDATE mail.box
       SET revision = current_revision,
           received_date = i_coords.received_date,
           newest_tif = i_newest_tif
     WHERE uid = i_uid
       AND mid = i_mid;
-- Updating pop3_box
    IF i_pop3initialized THEN
        UPDATE mail.pop3_box
           SET size = i_coords.size
         WHERE uid = i_uid
           AND mid = i_mid;
    END IF;
-- Update labels if folder is not trash
    IF i_msg_accounted_by_labels AND #i_lids > 0 THEN
        UPDATE mail.labels
           SET revision = current_revision
         WHERE uid = i_uid
           AND lid = ANY(i_lids);
    END IF;

-- Insert event into change log
    PERFORM impl.log_change(
        i_uid          => i_uid,
        i_request_info => i_request_info,
        i_revision     => current_revision,
        i_type         => 'quick-save',
        i_changed      => ((
            i_mid,
            i_fid,
            i_coords.st_id,
            i_coords.size)::impl.quick_saved_message
        )::impl.changed_with_fids
    );

    RETURN NEXT (true, current_revision)::code.quick_save_result;

END;
$$ LANGUAGE plpgsql;
