CREATE OR REPLACE FUNCTION code.resolve_labels(
    i_uid    code.uid,
    i_labels code.label_def[],
    i_request_info code.request_info DEFAULT NULL
) RETURNS SETOF code.label_info AS $$
DECLARE
    existed          code.label_info[];
    current_revision bigint;
    max_lid          integer;
    labels_created   impl.changed_label[];
BEGIN
    SELECT array_agg((lid, name, type, message_count, color, created, revision)::code.label_info)
      INTO existed
      FROM mail.labels
     WHERE uid = i_uid
       AND (name, type) IN (
        SELECT (l).name, (l).type
          FROM unnest(i_labels) l
    );

    IF existed IS NULL OR cardinality(existed) < cardinality(i_labels) THEN
        SELECT next_revision,
               next_lid
          INTO current_revision,
               max_lid
          FROM impl.acquire_current_serials(i_uid);

        WITH new_labels AS (
          INSERT INTO mail.labels
                 (uid, lid, revision,
                 name, type,
                 color, created)
          SELECT i_uid, max_lid + ll.rn - 1, current_revision,
                 ll.name, ll.type,
                 ll.color, created
            FROM (
              SELECT (l).name AS name, (l).type AS type,
                     (l).color AS color,
                     COALESCE((l).created, current_timestamp) AS created,
                     row_number() over() rn
                FROM unnest(i_labels) l
               WHERE NOT EXISTS (
                  SELECT 1 FROM mail.labels ild
                   WHERE ild.uid = i_uid
                     AND ild.name = (l).name
                     AND ild.type = (l).type)) ll
          RETURNING lid, name, type
        )
        SELECT array_agg((lid, name, type)::impl.changed_label)
          INTO labels_created
          FROM new_labels;

        IF labels_created IS NOT NULL AND cardinality(labels_created) > 0 THEN
            PERFORM impl.log_change(
                i_uid          => i_uid,
                i_request_info => i_request_info,
                i_revision     => current_revision,
                i_type         => 'label-create',
                i_changed      => to_jsonb(labels_created)
            );

            UPDATE mail.serials
               SET next_revision = next_revision + 1,
                   next_lid = next_lid + cardinality(labels_created)
             WHERE uid = i_uid;
        END IF;

        SELECT array_agg((lid, name, type, message_count, color, created, revision)::code.label_info)
          INTO existed
          FROM mail.labels
         WHERE uid = i_uid
           AND (name, type) IN (
              SELECT (l).name, (l).type
                FROM unnest(i_labels) l
          );
    END IF;

    RETURN QUERY SELECT * FROM unnest(existed);
END;
$$ LANGUAGE plpgsql;
