CREATE OR REPLACE FUNCTION code.sync_delete_messages(
    i_uid            code.uid,
    i_owner_uid      code.uid,
    i_owner_fid      code.fid,
    i_owner_mids     code.mids,
    i_owner_revision bigint,
    i_request_info   code.request_info DEFAULT NULL
) RETURNS code.operation_result AS $$
DECLARE
    v_subscription_id bigint := code.find_subscription_id_by_owner_uid_fid(i_uid, i_owner_uid, i_owner_fid);
    current_revision constant bigint := code.acquire_current_revision(i_uid);
    deleted_messages impl.moved_message[];
    deleted_sync_messages impl.sync_message_mapping[];
BEGIN
    WITH with_deleted AS (
        DELETE
          FROM mail.synced_messages
         WHERE uid = i_uid
           AND subscription_id = v_subscription_id
           AND owner_mid = ANY(i_owner_mids)
        RETURNING owner_mid, mid
    )
    SELECT array_agg((mid, owner_mid)::impl.sync_message_mapping)
      INTO deleted_sync_messages
      FROM with_deleted;

    IF deleted_sync_messages IS NULL OR cardinality(deleted_sync_messages) = 0 THEN
        RETURN code.unchanged_result();
    END IF;

    deleted_messages := impl.delete_messages_impl(
        i_uid,
        ARRAY(SELECT mid FROM unnest(deleted_sync_messages)),
        current_revision
    );

    PERFORM code.purge_deleted_messages(
        i_uid,
        ARRAY(SELECT mid FROM unnest(deleted_messages))
    );

    UPDATE mail.subscribed_folders
       SET synced_revision = i_owner_revision,
           revision = current_revision
     WHERE uid = i_uid
       AND subscription_id = v_subscription_id;

    PERFORM impl.log_change(
        i_uid => i_uid,
        i_request_info => i_request_info,
        i_revision => current_revision,
        i_type => 'sync-delete',
        i_changed => ARRAY(
            SELECT (i_owner_uid, owner_mid, i_owner_fid, mid, src_fid)::impl.synced_deleted_message
              FROM unnest(deleted_sync_messages) AS dsm
              JOIN unnest(deleted_messages) AS dm USING (mid)
        )::impl.changed_with_fids
    );

    PERFORM code.increment_revision(i_uid);

    RETURN (i_owner_revision, ARRAY(SELECT owner_mid FROM unnest(deleted_sync_messages)));
END;
$$ LANGUAGE plpgsql;
