CREATE OR REPLACE FUNCTION code.sync_join_threads(
    i_uid               code.uid,
    i_owner_uid         code.uid,
    i_owner_fid         code.fid,
    i_owner_tid         code.tid,
    i_owner_join_tids   code.tids,
    i_owner_revision    bigint,
    i_request_info      code.request_info DEFAULT NULL
) RETURNS code.operation_result AS $$
DECLARE
    v_subscription_id bigint := code.find_subscription_id_by_owner_uid_fid(i_uid, i_owner_uid, i_owner_fid);
    current_revision constant bigint := code.acquire_current_revision(i_uid);
    v_tid           bigint;
    v_synced_tids   bigint[];
    v_join_tids     bigint[];
    joined          impl.short_changed_message[];
    sync_messages   impl.sync_message_mapping[];
    v_unthreaded_mids bigint[];
    v_owner_tids_to_join bigint[] := array_append(i_owner_join_tids::bigint[], i_owner_tid::bigint);
BEGIN
    /* Resolving all owner merged tids into local tids*/
    SELECT array_agg(DISTINCT(tid)::bigint)
      INTO v_synced_tids
      FROM mail.synced_messages
     WHERE uid = i_uid
       AND subscription_id = v_subscription_id
       AND owner_tid = ANY(v_owner_tids_to_join);

    /* Selecting only tids for existing threads */
    SELECT array_agg(DISTINCT(tid)::bigint)
      INTO v_join_tids
      FROM mail.threads
     WHERE uid = i_uid
       AND tid = ANY(v_synced_tids);

    /*
    In case of all messages from the tread are placed into not threaded folder
    (e.g. Trash) the thread does not exists anymore. So we can not merge into
    nonexistent destination thread. What's why we need some logic to determine
    destination thread. This is the rule:
      - if no exiting threads found, tid of destination thread will be the
        original tid and no real merge will be performed, only mapping will
        be updated;
      - in other case we set as a destination first of the exiting threads.
    */
    IF coalesce(cardinality(v_join_tids), 0) = 0 THEN
        v_tid := code.find_synced_tid(i_uid, v_subscription_id, i_owner_tid);
    ELSE
        v_tid := v_join_tids[1];
    END IF;

    /*
    Real merge will be performed only if source treads are exist in other
    case - only mapping will be updated
    */
    IF coalesce(cardinality(v_join_tids), 0) > 1 THEN
        v_join_tids := v_join_tids[2:];
        joined := impl.join_threads(i_uid, v_tid, v_join_tids, current_revision);
    END IF;

    /*
    Perform update of threads mapping
    */
    WITH with_updated AS (
        UPDATE mail.synced_messages
           SET owner_tid = i_owner_tid,
               tid = v_tid,
               owner_revision = i_owner_revision
         WHERE uid = i_uid
           AND subscription_id = v_subscription_id
           AND owner_tid = ANY(v_owner_tids_to_join)
        RETURNING mid, owner_mid
    )
    SELECT array_agg((mid, owner_mid)::impl.sync_message_mapping)
      INTO sync_messages
      FROM with_updated;

    IF coalesce(cardinality(sync_messages), 0) = 0 THEN
        RETURN code.unchanged_result();
    END IF;

    /*
    UPDATE MESSAGES WITH RESETED TID
    There are may be messages moved to unthreaded folder, we need to change
    mail.messages.found_tid for them to make them able to restore in right
    tread.
    */
    SELECT array_agg(mid)
      INTO v_unthreaded_mids
      FROM mail.box mb JOIN unnest(sync_messages) USING (mid)
     WHERE uid = i_uid
       AND mb.tid IS NULL;

    UPDATE mail.messages
       SET found_tid = v_tid
     WHERE uid = i_uid
       AND mid = ANY(v_unthreaded_mids);

    /* Update revisions for subscribed folders */
    UPDATE mail.subscribed_folders
       SET synced_revision = i_owner_revision,
           revision = current_revision
     WHERE uid = i_uid
       AND subscription_id = v_subscription_id;

    PERFORM code.increment_revision(i_uid);

    PERFORM impl.log_change(
        i_uid          => i_uid,
        i_request_info => i_request_info,
        i_revision     => current_revision,
        i_type         => 'sync-threads-join',
        i_changed      => ARRAY(
            SELECT (i_owner_uid, smtbl.owner_mid, i_owner_fid,
                    jtbl.mid, jtbl.tid, jtbl.fid,
                    jtbl.seen, jtbl.recent, jtbl.deleted,
                    jtbl.lids)::impl.synced_short_changed_message
              FROM unnest(sync_messages) AS smtbl
              JOIN unnest(joined) AS jtbl USING (mid)
        )::impl.changed_with_fids,
        i_args         => jsonb_build_object(
            'tid', v_tid,
            'join_tids', v_join_tids,
            'owner_tid', i_owner_tid,
            'owner_join_tids', i_owner_join_tids)
    );

    RETURN (i_owner_revision, ARRAY(SELECT mid FROM unnest(joined)))::code.operation_result;
END;
$$ LANGUAGE plpgsql;
