CREATE OR REPLACE FUNCTION code.sync_update_messages(
    i_uid            code.uid,
    i_owner_uid      code.uid,
    i_owner_fid      code.fid,
    i_owner_mids     code.mids,
    i_owner_revision bigint,
    set_seen         boolean,
    set_recent       boolean,
    set_deleted      boolean,
    i_lids_add       integer[],
    i_lids_del       integer[],
    i_request_info   code.request_info DEFAULT NULL
) RETURNS code.operation_result AS $$
DECLARE
    v_subscription_id bigint := code.find_subscription_id_by_owner_uid_fid(i_uid, i_owner_uid, i_owner_fid);
    updated_sync_messages impl.sync_message_mapping[];
    updated_messages      impl.updated_message[];
    current_revision      constant bigint := code.acquire_current_revision(i_uid);
    lids_add              integer[] := uniq(sort(coalesce(i_lids_add, ARRAY[]::integer[])));
    lids_del              integer[] := uniq(sort(coalesce(i_lids_del, ARRAY[]::integer[])));
BEGIN
    SELECT array_agg((mid, owner_mid)::impl.sync_message_mapping)
      INTO updated_sync_messages
      FROM mail.synced_messages
     WHERE uid = i_uid
       AND subscription_id = v_subscription_id
       AND owner_mid = ANY(i_owner_mids);

    IF updated_sync_messages IS NULL OR cardinality(updated_sync_messages) = 0 THEN
        RETURN code.unchanged_result();
    END IF;

    updated_messages := impl.update_messages_impl(
        i_uid,
        ARRAY(SELECT mid FROM unnest(updated_sync_messages)),
        set_seen, set_recent, set_deleted,
        lids_add, lids_del, current_revision
    );

    IF updated_messages IS NULL OR cardinality(updated_messages) = 0 THEN
        RETURN code.unchanged_result();
    END IF;

    UPDATE mail.subscribed_folders
       SET synced_revision = i_owner_revision,
           revision = current_revision
     WHERE uid = i_uid
       AND subscription_id = v_subscription_id;

    PERFORM impl.log_change(
        i_uid => i_uid,
        i_request_info => i_request_info,
        i_revision => current_revision,
        i_type => 'sync-update',
        i_changed => ARRAY(
            SELECT (i_owner_uid, owner_mid, i_owner_fid, mid, tid, fid,
                    new_seen, new_recent, new_deleted, new_lids)::impl.synced_short_changed_message
              FROM unnest(updated_sync_messages) AS usm
              JOIN unnest(updated_messages) AS um USING (mid)
        )::impl.changed_with_fids,
        i_args         => jsonb_build_object(
            'seen',     set_seen,
            'recent',   set_recent,
            'deleted',  set_deleted,
            'lids_add', lids_add,
            'lids_del', lids_del
        )
    );

    PERFORM code.increment_revision(i_uid);

    RETURN (i_owner_revision, ARRAY(SELECT owner_mid FROM unnest(updated_sync_messages)));
END;
$$ LANGUAGE plpgsql;
