CREATE OR REPLACE FUNCTION code.update_label(
    i_uid   bigint,
    i_lid   integer,
    i_name  text,
    i_color text,
    i_request_info code.request_info DEFAULT NULL
) RETURNS mail.labels AS $$
DECLARE
    current_revision bigint := code.acquire_current_revision(i_uid);
    old_name         text;
    old_color        text;
    label            mail.labels%ROWTYPE;
    changed          json := json_build_object();
BEGIN
    SELECT name, color
      FROM mail.labels
     WHERE uid = i_uid
       AND lid = i_lid
       FOR UPDATE
      INTO old_name, old_color;

    IF NOT found THEN
        RAISE EXCEPTION 'No label updated uid: %, lid: %' , i_uid, i_lid;
    END IF;

    UPDATE mail.labels
       SET name = i_name,
           color = i_color,
           revision = current_revision
     WHERE uid = i_uid
       AND lid = i_lid
    RETURNING *
    INTO label;

    IF old_name != i_name THEN
        changed := impl.json_merge(changed, json_build_object('name', i_name));
    END IF;

    IF COALESCE(old_color, 'None') != COALESCE(i_color, 'None') THEN
        changed := impl.json_merge(changed, json_build_object('color', i_color));
    END IF;

    PERFORM impl.log_change(
        i_uid          => i_uid,
        i_request_info => i_request_info,
        i_revision     => current_revision,
        i_type         => 'label-modify',
        i_changed      => jsonb_build_array(changed::jsonb),
        i_args         => json_build_object('lid', i_lid)::jsonb
    );

    PERFORM code.increment_revision(i_uid);

    RETURN label;
END;
$$ LANGUAGE plpgsql;
