CREATE OR REPLACE FUNCTION code.register_user(
    i_uid                code.uid,
    i_country            text,
    i_lang               text,
    i_need_welcomes      boolean,
    i_request_info       code.request_info DEFAULT NULL,
    i_need_filters       boolean DEFAULT FALSE
) RETURNS code.registration_result AS $$
DECLARE
    c_inbox_fid          integer;
    init_revision        bigint;
    v_user               mail.users;
    request_id           text;
    contacts_user_creating_result code.create_contacts_user_result;
BEGIN
    IF NOT pg_try_advisory_xact_lock(i_uid) THEN
        RETURN 'already_in_progress';
    END IF;

    SELECT *
      INTO v_user
      FROM mail.users
     WHERE uid = i_uid;

    IF FOUND THEN
        IF NOT v_user.is_here THEN
            RETURN 'shard_is_occupied_by_user';
        ELSE
            IF NOT v_user.is_deleted THEN
                RETURN 'already_registered';
            END IF;
        END IF;

        PERFORM code.reinit_deleted_user(i_uid);
    ELSE
        PERFORM code.init_user(i_uid);

        IF i_need_filters THEN
            PERFORM impl.create_base_filters(i_uid);
        END IF;
    END IF;

    request_id = COALESCE(CASE WHEN i_request_info IS NOT NULL THEN i_request_info.x_request_id END,
        'register_user');
    contacts_user_creating_result := code.create_contacts_user(i_uid, 'passport_user', request_id);
    IF contacts_user_creating_result <> 'success' THEN
        RAISE WARNING 'Execution of code.create_contacts_user for uid = % (request_id = %) returned ''%''',
            i_uid, request_id, contacts_user_creating_result;
    END IF;

    init_revision := code.increment_revision(i_uid);

    PERFORM impl.log_change(
        i_uid,
        i_request_info,
        init_revision,
        'register',
        (0, 0)::impl.changelog_counters,
        NULL::jsonb,
        json_build_object(
            'country', i_country,
            'lang', i_lang,
            'need_welcomes', i_need_welcomes
        )::jsonb
    );

    IF i_need_welcomes THEN
        c_inbox_fid := code.get_folder_fid(i_uid, 'inbox');

        PERFORM code.store_welcome_messages(i_uid, c_inbox_fid, i_country, i_lang, i_request_info,
                                            'relevant'::mail.tab_types);
    END IF;
    RETURN 'success';
END;
$$ LANGUAGE plpgsql;
