CREATE OR REPLACE FUNCTION code.sync_message(
    i_uid           code.uid,
    i_owner_coords  code.owner_coordinates,
    i_coords        code.sync_coordinates,
    i_headers       code.sync_headers,
    i_recipients    code.sync_recipient[],
    i_attaches      code.sync_attach[],
    i_sync_lids     code.unique_lids,
    i_threads       code.sync_threading,
    i_request_info  code.request_info DEFAULT NULL,
    i_mime          code.sync_mime_part[] DEFAULT NULL,
    i_quiet         boolean DEFAULT NULL,
    i_imap_id       bigint DEFAULT NULL
) RETURNS SETOF code.sync_result AS $$
DECLARE
    v_subscription_id bigint := code.find_subscription_id_by_owner_uid_fid(
            i_uid, (i_owner_coords).uid, (i_owner_coords).fid);
    v_store_res      impl.store_result;
    v_store_coords   impl.store_coordinates;
    v_synced_row     mail.synced_messages;
    v_subscribed_fid int;
    v_subscribed_tid bigint;
BEGIN
    SELECT fid
      INTO v_subscribed_fid
      FROM mail.subscribed_folders
     WHERE uid = i_uid
       AND subscription_id = v_subscription_id;

    IF NOT found THEN
        RAISE EXCEPTION 'User % do not have subscribed_folders from owner: <%,%>',
                         i_uid, i_owner_coords.uid, i_owner_coords.fid
           USING SCHEMA = 'mail',
                  TABLE = 'subscribed_folders';
    END IF;

    SELECT *
      INTO v_synced_row
      FROM mail.synced_messages
     WHERE uid = i_uid
       AND subscription_id = v_subscription_id
       AND owner_mid = (i_owner_coords).mid;

    IF NOT found THEN
        -- lock user and lookup again
        -- cause we may call synced_message concurrently
        PERFORM code.acquire_current_revision(i_uid);

        SELECT *
          INTO v_synced_row
          FROM mail.synced_messages
         WHERE uid = i_uid
           AND subscription_id = v_subscription_id
           AND owner_mid = (i_owner_coords).mid;
    END IF;

    IF NOT found THEN
        -- don't need to care about tid IS NULL
        -- impl.store_message treat it as new thread
        v_subscribed_tid := code.find_synced_tid(
            i_uid       => i_uid,
            i_subscription_id => v_subscription_id,
            i_owner_tid => i_owner_coords.tid
        );
        v_store_coords := code.cast_code_sync_coordinates_to_impl(
            i_subscribed_fid => v_subscribed_fid,
            i_subscribed_tid => v_subscribed_tid,
            i_coords         => i_coords
        );

        v_store_res := impl.store_message(
            i_uid          => i_uid,
            i_coords       => v_store_coords,
            i_headers      => i_headers::impl.store_headers,
            i_recipients   => i_recipients::mail.recipient[],
            i_attaches     => i_attaches::mail.attach[],
            i_mime         => i_mime::mail.mime_part[],
            i_lids         => i_sync_lids,
            i_threads      => i_threads::impl.store_threading
        );

        INSERT INTO mail.synced_messages
            (uid, mid,
             tid, revision,
             subscription_id, owner_mid,
             owner_tid, owner_revision)
        VALUES
            (i_uid, v_store_res.mid,
             v_store_res.tid, v_store_res.revision,
             v_subscription_id, i_owner_coords.mid,
             i_owner_coords.tid, i_owner_coords.revision)
        RETURNING * INTO v_synced_row;

        UPDATE mail.subscribed_folders
           SET synced_revision = GREATEST(i_owner_coords.revision, synced_revision),
               revision = (v_store_res).revision,
               synced_imap_id = i_imap_id
         WHERE uid = i_uid
           AND fid = v_subscribed_fid;

        PERFORM impl.log_change(
            i_uid          => i_uid,
            i_request_info => i_request_info,
            i_revision     => v_store_res.revision,
            i_type         => 'sync-store',
            i_changed => (
                (i_owner_coords.uid, i_owner_coords.mid, i_owner_coords.fid,
                 v_store_res.mid, v_store_res.tid, v_store_res.fid,
                 v_store_res.seen, v_store_res.recent, v_store_res.deleted,
                 v_store_res.lids, v_store_res.hdr_message_id)::impl.synced_changed_message
            )::impl.changed_with_fids,
            i_counters     => (
                v_store_res.fresh_count,
                impl.get_useful_count(i_uid))::impl.changelog_counters,
            i_quiet        => i_quiet
        );
    END IF;

    RETURN NEXT (
      v_synced_row.mid,
      v_synced_row.revision,
      v_synced_row.owner_mid,
      v_synced_row.owner_revision
    );
END;
$$ LANGUAGE plpgsql;
