CREATE OR REPLACE FUNCTION code.create_collector(
    i_dst_uid      code.uid,
    i_src_uid      code.uid,
    i_auth_token   text,
    i_root_folder  integer DEFAULT NULL,
    i_label        integer DEFAULT NULL,
    i_request_info code.request_info DEFAULT NULL
) RETURNS code.collector_id_result AS $$
DECLARE
    current_revision bigint;
    new_collector_id integer;
    src_uid_exists boolean;
BEGIN
    SELECT next_revision,
           next_collector_id
      INTO current_revision,
           new_collector_id
    FROM impl.acquire_current_serials(i_dst_uid);

    IF i_dst_uid = i_src_uid THEN
        RAISE EXCEPTION 'Can''t create collector from myself, uid: %', i_dst_uid
            USING TABLE = 'mail.collectors';
    END IF;

    SELECT EXISTS(
        SELECT 1
          FROM mail.collectors
         WHERE uid = i_dst_uid
           AND (metadata->>'src_uid')::bigint = i_src_uid)
      INTO src_uid_exists;

    IF src_uid_exists THEN
        RAISE EXCEPTION 'Can''t create collector - already exists (dst_uid: %, src_uid: %)', i_dst_uid, i_src_uid
            USING TABLE = 'mail.collectors';
    END IF;

    INSERT INTO mail.collectors
        (uid, collector_id, metadata)
    VALUES (i_dst_uid, new_collector_id,
            jsonb_build_object('src_uid', i_src_uid,
                               'auth_token', i_auth_token,
                               'creation_ts', current_timestamp,
                               'root_folder_id', i_root_folder,
                               'label_id', i_label));

    PERFORM impl.log_change(
        i_uid          => i_dst_uid,
        i_request_info => i_request_info,
        i_revision     => current_revision,
        i_type         => 'collector-create',
        i_changed      => jsonb_build_array(
                              jsonb_build_object('src_uid', i_src_uid,
                                                 'root_folder_id', i_root_folder,
                                                 'label_id', i_label)));

    UPDATE mail.serials
       SET next_revision = next_revision + 1,
           next_collector_id = next_collector_id + 1
     WHERE uid = i_dst_uid;

    RETURN (current_revision, new_collector_id);
END;
$$ LANGUAGE plpgsql;
