CREATE OR REPLACE FUNCTION code.regenerate_imap_id (
    i_uid   code.uid,
    i_mid   code.mid,
    i_request_info code.request_info DEFAULT NULL
) RETURNS code.imap_id_result AS $$
DECLARE
    new_imap_id bigint;
    last_chained impl.chained_message;
    current_revision bigint := code.acquire_current_revision(i_uid);
    moved impl.moved_message;
BEGIN
    SELECT mb.mid,
           mb.fid, mb.fid,
           mb.tid, mb.tid,
           0, mb.imap_id,
           mb.chain,
           mb.seen, mb.recent, mb.deleted,
           mb.lids,
           mm.size,
           mb.received_date,
           mb.newest_tif,
           impl.get_attaches_info(mm.attaches),
           mm.hdr_message_id,
           mm.attributes
      INTO moved
      FROM mail.box mb
      JOIN mail.messages mm
        ON (mb.mid = mm.mid and mm.uid = i_uid)
     WHERE mb.mid = i_mid
       AND mb.uid = i_uid;

    IF NOT found THEN
        RAISE EXCEPTION 'Nonexistent message mid: %, uid: %', i_mid, i_uid
              USING HINT = 'Please check your user ID',
                    TABLE = 'mail.box';
    END IF;

    last_chained := impl.get_last_chained(i_uid, moved.src_fid);

    SELECT next_imap_id
      INTO new_imap_id
      FROM mail.folders
     WHERE uid = i_uid
       AND fid = moved.src_fid;

    moved.dst_imap_id := new_imap_id;

    UPDATE mail.box m
       SET revision = current_revision,
           chain = CASE WHEN moved.src_chain is NOT NULL AND last_chained.imap_id = moved.src_imap_id
                        THEN 1
                        ELSE impl.new_chain_formula(1, 1, last_chained) END,
           imap_id = new_imap_id,
           recent = true
     WHERE uid = i_uid
       AND mid = i_mid;

    UPDATE mail.folders
       SET next_imap_id = next_imap_id + 1,
           revision = current_revision,
           first_unseen = (counters).first_unseen,
           first_unseen_id = (counters).first_unseen_id,
           message_recent = message_recent + 1 - moved.recent::integer
      FROM (
        SELECT CASE WHEN first_unseen_id IS NULL -- no unseen messages in folder
                      OR first_unseen_id < moved.src_imap_id -- no messages moved before first_useen
                    THEN (first_unseen, first_unseen_id)::impl.first_unseen_def
                    ELSE impl.find_first_unseen(i_uid, moved.src_fid) END AS counters
          FROM mail.folders
         WHERE uid = i_uid
           AND fid = moved.src_fid) fi
     WHERE uid = i_uid
       AND fid = moved.src_fid;

    IF moved.src_chain is NULL OR last_chained.imap_id != moved.src_imap_id THEN
        PERFORM impl.update_chains_for_added_messages(
            i_uid,
            moved.src_fid,
            array_agg(moved),
            last_chained
        );
    END IF;

    PERFORM impl.update_chains_for_removed_messages(
        i_uid,
        current_revision,
        array_agg(moved)
    );

    PERFORM impl.log_change(
        i_uid          => i_uid,
        i_request_info => i_request_info,
        i_revision     => current_revision,
        i_type         => 'reindex',
        i_changed      => json_agg(json_build_object('mid', i_mid))::jsonb
    );

    PERFORM code.increment_revision(i_uid);

    RETURN (current_revision, new_imap_id);
END;
$$ LANGUAGE plpgsql;
