DROP SCHEMA IF EXISTS filters CASCADE;
CREATE SCHEMA filters;

CREATE SEQUENCE filters.rule_seq
    START WITH 1 INCREMENT BY 1 NO MAXVALUE;
CREATE SEQUENCE filters.action_seq
    START WITH 1 INCREMENT BY 1 NO MAXVALUE;
CREATE SEQUENCE filters.condition_seq
    START WITH 1 INCREMENT BY 1 NO MAXVALUE;

CREATE TYPE filters.rule_type AS ENUM (
    'user', 'unsubscribe', 'system'
);

CREATE TABLE filters.rules (
    uid     bigint NOT NULL,
    rule_id bigint NOT NULL,
    name    text,
    enabled boolean NOT NULL DEFAULT TRUE,
    prio    bigint NOT NULL DEFAULT 0,
    stop    boolean NOT NULL DEFAULT FALSE,
    created timestamp with time zone NOT NULL DEFAULT current_timestamp,
    type    filters.rule_type NOT NULL DEFAULT 'user',

    CONSTRAINT pk_rules PRIMARY KEY (uid, rule_id),
    CONSTRAINT fk_rules_mail_users FOREIGN KEY (uid)
        REFERENCES mail.users (uid) ON DELETE NO ACTION
);

CREATE TYPE filters.action_oper_type AS ENUM (
    'move', 'delete', 'movel', 'status',
    'forward', 'forwardwithstore', 'reply', 'notify'
);

CREATE TABLE filters.actions (
    uid       bigint NOT NULL,
    rule_id   bigint NOT NULL,
    action_id bigint NOT NULL DEFAULT nextval('filters.action_seq'),
    oper      filters.action_oper_type NOT NULL,
    param     text,
    verified  boolean NOT NULL DEFAULT TRUE,

    CONSTRAINT pk_actions PRIMARY KEY (uid, action_id),
    CONSTRAINT fk_rules FOREIGN KEY (uid, rule_id)
        REFERENCES filters.rules (uid, rule_id) ON DELETE CASCADE
);

CREATE INDEX i_actions_uid_rule ON filters.actions
    USING btree (uid, rule_id);

CREATE TYPE filters.condition_field_type AS ENUM (
    'header', 'body', 'attach', 'flag', 'type'
);

CREATE TYPE filters.condition_oper_type AS ENUM (
    'contains', 'matches', 'exists'
);

CREATE TYPE filters.condition_link_type AS ENUM (
    'or', 'and'
);

CREATE TABLE filters.conditions (
    uid          bigint NOT NULL,
    rule_id      bigint NOT NULL,
    field_type   filters.condition_field_type NOT NULL DEFAULT 'header',
    field        text,
    pattern      text,
    oper         filters.condition_oper_type NOT NULL DEFAULT 'contains',
    link         filters.condition_link_type NOT NULL DEFAULT 'or',
    negative     boolean NOT NULL DEFAULT FALSE,
    condition_id bigint NOT NULL DEFAULT nextval('filters.condition_seq'),

    CONSTRAINT pk_conditions PRIMARY KEY (uid, condition_id),
    CONSTRAINT fk_rules FOREIGN KEY (uid, rule_id)
        REFERENCES filters.rules (uid, rule_id) ON DELETE CASCADE
);

CREATE INDEX i_conditions_uid_rule ON filters.conditions USING btree (uid, rule_id);

CREATE TYPE filters.elist_type AS ENUM (
    'black', 'white'
);

CREATE TABLE filters.elist (
    uid     bigint NOT NULL,
    email   text NOT NULL,
    list    filters.elist_type NOT NULL DEFAULT 'black',
    created timestamp with time zone NOT NULL DEFAULT current_timestamp,

    CONSTRAINT pk_elist PRIMARY KEY (uid, email),
    CONSTRAINT fk_elist_mail_users FOREIGN KEY (uid)
        REFERENCES mail.users (uid) ON DELETE NO ACTION
);
