CREATE OR REPLACE FUNCTION impl.update_chains_for_removed_messages(
    i_uid            bigint,
    current_revision bigint,
    moved         impl.moved_message[]
) RETURNS void AS $$
DECLARE
    c       record;
    i_moved impl.moved_message;

    new_chained         bigint[];
    new_chained_imap_id bigint;
BEGIN
    /* update chains in old folders */
    FOR c IN (
        SELECT ic.*, f.message_count, f.next_imap_id
          FROM (
            SELECT (um).src_fid AS src_fid,
                   array_agg(um)
                      FILTER (WHERE (um).src_chain IS NULL) AS unchained_moved,
                   /* ORDER cause we iterate over c.chained_moved */
                   array_agg(um ORDER BY um.src_imap_id)
                      FILTER (WHERE (um).src_chain IS NOT NULL) AS chained_moved,
                   array_agg(um) AS per_folder_moved
              FROM unnest(moved) um
             GROUP BY (um).src_fid) ic
          JOIN mail.folders f
            ON (ic.src_fid = f.fid AND f.uid = i_uid)
    ) LOOP
        /*
          While save to chained_log, calculate distance
          (aka message count between chained and moved)
          base on old chains
        */
        WITH chains_intervals AS (
            SELECT chain_start_id,
                   (SELECT array_agg(um)
                      FROM unnest(c.per_folder_moved) um
                     WHERE um.src_imap_id >= chain_start_id
                       AND (
                        chain_end_id IS NULL
                        OR
                        um.src_imap_id < chain_end_id)
                   ) AS per_chain_moved
              FROM (
                SELECT imap_id AS chain_start_id,
                       lead(imap_id) over (ORDER BY imap_id) AS chain_end_id
                  FROM (
                    SELECT imap_id
                      FROM mail.box
                     WHERE uid = i_uid
                       AND fid = c.src_fid
                       AND chain IS NOT NULL
                     UNION ALL
                    SELECT (cm).src_imap_id
                      FROM unnest(c.chained_moved) cm) ci
                ) cd)
        INSERT INTO mail.chained_log
            (uid, fid, revision, mid,
            chained_id, imap_id,
            distance)
        SELECT
            i_uid, c.src_fid, current_revision, (uc).mid,
            chain_start_id, (uc).src_imap_id,
            (SELECT count(*) -- actual message count
               FROM mail.box m
              WHERE uid = i_uid
                AND fid = c.src_fid
                AND imap_id >= chain_start_id
                AND imap_id < (uc).src_imap_id) +
            (SELECT count(*) -- moved message count
               FROM unnest(ci.per_chain_moved) mc
              WHERE (mc).src_imap_id <= (uc).src_imap_id)
          FROM chains_intervals ci,
               unnest(ci.per_chain_moved) uc;

        IF NOT found THEN
            RAISE EXCEPTION
                'Got itself in trouble src_fid %, '
                ' per_folder_moved %, chained_moved %',
                    c.src_fid, c.per_folder_moved, c.chained_moved;
        END IF;

        IF c.message_count = 0 THEN
            CONTINUE;
        END IF;

        -- setup new chains for chained moved
        FOR i IN 1 .. coalesce(cardinality(c.chained_moved), 0)
        LOOP
            i_moved := c.chained_moved[i];
            -- chained contains only itself
            IF i_moved.src_chain = 1 THEN
                CONTINUE;
            END IF;
            -- find previous chained and update it's chain
            UPDATE mail.box m
               SET chain = chain + i_moved.src_chain - 1
              FROM (
                  SELECT imap_id AS prev_chained_imap_id
                    FROM mail.box
                   WHERE uid = i_uid
                     AND fid = c.src_fid
                     AND chain IS NOT NULL
                     AND imap_id < i_moved.src_imap_id
                   ORDER BY imap_id DESC
                   LIMIT 1
                ) pm
             WHERE uid = i_uid
               AND m.fid = c.src_fid
               AND m.imap_id = pm.prev_chained_imap_id
            RETURNING imap_id INTO new_chained_imap_id;

            IF NOT found THEN
                /* probably we move first chained in folder
                   there are 2 cases:
                      1. we remove chained with it's interval - don't need to do something special,
                         but we can't simply define this case
                      2. we remove chained, but not all messages from it's interval - we need
                         to set need first chained as min(imap_id) in src_fid
                */
                IF i_moved.src_chain != 1 THEN -- this chained include it's interval
                    UPDATE mail.box
                       SET chain = coalesce(chain, 0)
                                   + i_moved.src_chain -- add previous chain
                                   - pm.moved_chain
                                   - 1 -- @i_moved is moved
                      FROM (
                          SELECT min_imap_id,
                                 (SELECT count(1)
                                    FROM unnest(c.unchained_moved) um
                                   WHERE um.src_imap_id < mm.min_imap_id
                                     AND um.src_imap_id > i_moved.src_imap_id) AS moved_chain
                            FROM (
                              SELECT min(imap_id) AS min_imap_id
                                FROM mail.box
                               WHERE uid = i_uid
                                 AND fid = c.src_fid
                                 AND imap_id < coalesce(
                                    c.chained_moved[i+1].src_imap_id, -- SQL magic
                                    c.next_imap_id)
                              ) mm
                        ) pm
                     WHERE uid = i_uid
                       AND fid = c.src_fid
                       AND imap_id = pm.min_imap_id
                     RETURNING imap_id INTO new_chained_imap_id;
                END IF;
            END IF;
            IF new_chained_imap_id IS NOT NULL THEN
                new_chained := array_append(new_chained, new_chained_imap_id);
            END IF;
        END LOOP;
        -- we move some unchained message
        IF cardinality(c.unchained_moved) > 0 THEN
            -- iterate over all chained in folder
            UPDATE mail.box m
               SET chain = chain - moved_chain
              FROM (
                SELECT chain_start_id,
                       (SELECT count(1)
                          FROM unnest(c.unchained_moved) um
                         WHERE um.src_imap_id > chain_start_id
                           AND (
                            chain_end_id IS NULL
                            OR
                            um.src_imap_id < chain_end_id)
                       ) AS moved_chain
                  FROM (
                    SELECT imap_id AS chain_start_id,
                           lead(imap_id) over (ORDER BY imap_id) AS chain_end_id
                      FROM mail.box
                     WHERE uid = i_uid
                       AND fid = c.src_fid
                       AND chain IS NOT NULL
                    ) cd
                ) mc
            WHERE uid = i_uid
              AND fid = c.src_fid
              AND moved_chain > 0 -- need explain, maybe move it's filter to subquery
              AND imap_id = chain_start_id;
        END IF;
        /* for each new chained in folder we save:
            * it's imap_id as chained_id
            * 0 as imap_id -- it's a marker for chained change
            * it's chain as distance
        */
        IF cardinality(new_chained) > 0 THEN
            INSERT INTO mail.chained_log
                (uid, fid, revision, mid,
                chained_id, imap_id,
                distance)
            SELECT i_uid, c.src_fid, current_revision, mid,
                imap_id, 0,
                chain
              FROM mail.box
             WHERE uid = i_uid
               AND fid = c.src_fid
               AND chain IS NOT NULL
               AND imap_id = ANY(new_chained);
        END IF;
     END LOOP;
END;
$$ LANGUAGE plpgsql;
