CREATE OR REPLACE FUNCTION impl.is_countable_fid(
    i_uid  bigint,
    i_fid  integer
) RETURNS  boolean AS $$
BEGIN
    RETURN COALESCE(NOT impl.is_hidden_folder(impl.get_folder_type(i_uid, i_fid)), false);
END;
$$ LANGUAGE plpgsql;


CREATE OR REPLACE FUNCTION impl.update_counters(
    i_uid               bigint,
    i_current_revision  bigint,
    i_moved             impl.moved_message[]
) RETURNS void AS $$
DECLARE
    v_has_attach_count  integer;
    v_has_attach_seen   integer;
BEGIN
    /*
        Get counter changes considering that:
        1. We only need to update counters for messages with attaches
        2. Moving from src_fid == dst_fid does not affect counters (whatever folder types are)
        3. Moving from not hidden folder should decrease counters
        4. Moving to not hidden folder should increase counters
        5. Moving from not hidden to not hidden should not affect counters
            - this will be automatically provided by 3 and 4
    */
    SELECT COALESCE(sum(att_inc::integer - att_dec::integer), 0),
           COALESCE(sum((att_inc AND seen)::integer - (att_dec AND seen)::integer), 0)
      INTO v_has_attach_count, v_has_attach_seen
      FROM (
        SELECT impl.is_countable_fid(i_uid, m.src_fid) AS att_dec,
               impl.is_countable_fid(i_uid, m.dst_fid) AS att_inc,
               m.seen AS seen
          FROM unnest(i_moved) m
         WHERE m.src_fid IS DISTINCT FROM m.dst_fid
           AND (m.ai).attach_count > 0 ) att ;

    /*
        For old users new attach counters columns will be equal to 0.
        If they delete some of messages with attaches, we will try to decrease counters.
        But we cannot set negative attach counters, so we use greatest(val, 0) here.
    */
    IF v_has_attach_count != 0 OR v_has_attach_seen != 0 THEN
        UPDATE mail.counters
           SET has_attaches_count = GREATEST(has_attaches_count + v_has_attach_count, 0),
               has_attaches_seen = GREATEST(has_attaches_seen + v_has_attach_seen, 0),
               revision = i_current_revision
         WHERE uid = i_uid;
    END IF;
END;
$$ LANGUAGE plpgsql;
