CREATE OR REPLACE FUNCTION impl.update_dst_folder(
    i_uid            bigint,
    i_dst_fid        integer,
    current_revision bigint,
    moved            impl.moved_message[],
    last_chained     impl.chained_message
) RETURNS void AS $$
DECLARE
    c  record;
    prev_newest_exists boolean;
    update_prev_newest boolean;
    prev_newest_mid    bigint;
    v_real_moved       impl.moved_message[];
BEGIN
    /*
        1. update folders revision
        2. get real moved
        3. update folders counters
        4. update mail.box chains
        5. update newest_tif
    */

    -- Update folders revision
    UPDATE mail.folders
       SET revision = current_revision
     WHERE uid = i_uid
       AND fid = i_dst_fid;

    -- get real moved
    SELECT ARRAY (
        SELECT m
          FROM unnest(moved) m
         WHERE m.src_fid IS DISTINCT FROM i_dst_fid )
    INTO v_real_moved;

    IF v_real_moved IS NULL OR cardinality(v_real_moved) = 0 THEN
        RETURN;
    END IF;

    -- Update folders counters
    UPDATE mail.folders
       SET message_count = message_count + moved_count,
           message_recent = message_recent + moved_count,
           message_seen = message_seen + moved_seen,
           message_size = message_size + moved_size,
           attach_count = attach_count + moved_attach_count,
           attach_size = attach_size + moved_attach_size,
           first_unseen =
              CASE
              WHEN first_unseen_id IS NULL
               AND moved_first_unseen_id IS NOT NULL
              THEN message_count -- offset of first moved imap_id
                   -- add message count between first moved and moved_first_unseen_id
                   + moved_first_unseen_id
                   - next_imap_id
                   + 1
               ELSE first_unseen END,
           first_unseen_id = coalesce(first_unseen_id, moved_first_unseen_id),
           next_imap_id = next_imap_id + moved_count
       FROM (
        SELECT count(*) AS moved_count,
               sum(m.seen::integer) AS moved_seen,
               sum(m.size::bigint) AS moved_size,
               sum((ai).attach_count) AS moved_attach_count,
               sum((ai).attach_size) AS moved_attach_size,
               min(moved_first_unseen_id) AS moved_first_unseen_id
          FROM (
            SELECT im.*,
                   min(dst_imap_id)
                      FILTER (WHERE NOT seen)
                      OVER (PARTITION BY seen ORDER BY dst_imap_id)
                   AS moved_first_unseen_id
              FROM unnest(v_real_moved) im) m) fi
      WHERE uid = i_uid
        AND fid = i_dst_fid;

    PERFORM impl.update_chains_for_added_messages(i_uid, i_dst_fid, v_real_moved, last_chained);

    /* update newest_tif */
    FOR c IN (
        SELECT dst_tid,
               min(newest_moved_date) AS newest_moved_date,
               min(newest_moved_mid) AS newest_moved_mid
          FROM (
            SELECT m.*,
                   last_value(received_date) over mids_in_thread AS newest_moved_date,
                   last_value(mid) over mids_in_thread AS newest_moved_mid
              FROM unnest(v_real_moved) m
             WHERE m.dst_tid IS NOT NULL
            WINDOW mids_in_thread AS (
                   PARTITION BY dst_tid
                   ORDER BY received_date
                   ROWS
                      BETWEEN UNBOUNDED PRECEDING
                          AND UNBOUNDED FOLLOWING)
               ) m
         GROUP BY dst_tid)
    LOOP
        SELECT mid, received_date < c.newest_moved_date
          INTO prev_newest_mid, update_prev_newest
          FROM mail.box
         WHERE uid = i_uid
           AND fid = i_dst_fid
           AND tid = c.dst_tid
           AND newest_tif = true;
        prev_newest_exists := found;
        IF NOT prev_newest_exists OR update_prev_newest THEN
            UPDATE mail.box
               SET newest_tif = true
             WHERE uid = i_uid
               AND mid = c.newest_moved_mid;
        END IF;
        IF prev_newest_exists AND update_prev_newest THEN
            UPDATE mail.box
               SET newest_tif = false
             WHERE uid = i_uid
               AND mid = prev_newest_mid;
        END IF;
    END LOOP;
END;
$$ LANGUAGE plpgsql;