CREATE OR REPLACE FUNCTION impl.update_src_folders(
    i_uid            bigint,
    current_revision bigint,
    moved            impl.moved_message[]
) RETURNS void AS $$
DECLARE
    v_real_moved     impl.moved_message[];
BEGIN
    /*
        1. update folders revision
        2. get real moved
        3. update folders counters
        4. update mail.box chains
        5. update newest_tif
    */

    -- update folders revision
    UPDATE mail.folders f
       SET revision = current_revision
       FROM (
        SELECT fd.*
           FROM (
               SELECT DISTINCT(src_fid)
                 FROM unnest(moved) m
           ) fd
       JOIN mail.folders fi
         ON (fi.uid = i_uid AND fi.fid = src_fid) ) fi
      WHERE uid = i_uid
        AND f.fid = src_fid;

    -- get real moved
    SELECT ARRAY (
        SELECT m
          FROM unnest(moved) m
         WHERE m.src_fid IS DISTINCT FROM m.dst_fid )
    INTO v_real_moved;

    IF v_real_moved IS NULL OR cardinality(v_real_moved) = 0 THEN
        RETURN;
    END IF;

    -- update folders counters
    UPDATE mail.folders f
       SET message_count = message_count - moved_count,
           message_recent = message_recent - moved_recent,
           message_seen = message_seen - moved_seen,
           message_size = message_size - moved_size,
           attach_count = attach_count - moved_attach_count,
           attach_size = attach_size - moved_attach_size,
           first_unseen = (fd).first_unseen,
           first_unseen_id = (fd).first_unseen_id
       FROM (
        SELECT fd.*,
               CASE WHEN first_unseen_id IS NULL -- no unseen messages in folder
                      OR first_unseen_id < min_moved_imap_id -- no messages moved before first_useen
                        THEN (first_unseen, first_unseen_id)::impl.first_unseen_def
                    WHEN message_count = moved_count
                        THEN (0, null)::impl.first_unseen_def
                    ELSE impl.find_first_unseen(i_uid, src_fid, first_unseen, first_unseen_id, v_real_moved) END AS fd
          FROM (
            SELECT src_fid,
                   count(*) AS moved_count,
                   sum(m.seen::integer) AS moved_seen,
                   sum(m.recent::integer) AS moved_recent,
                   sum(m.size::bigint) AS moved_size,
                   sum((ai).attach_count) AS moved_attach_count,
                   sum((ai).attach_size) AS moved_attach_size,
                   min(src_imap_id) AS min_moved_imap_id
              FROM unnest(v_real_moved) m
             GROUP BY src_fid) fd
          JOIN mail.folders fi
            ON (fi.uid = i_uid AND fi.fid = src_fid) ) fi
      WHERE uid = i_uid
        AND f.fid = src_fid;

    PERFORM impl.update_chains_for_removed_messages(i_uid, current_revision, v_real_moved);

    -- update newest_tif for src folders
    UPDATE mail.box
       SET newest_tif = true
      FROM (
        SELECT mm.mid AS new_newest_mid
          FROM (
            SELECT src_fid, src_tid
              FROM (
                SELECT src_fid, src_tid
                  FROM unnest(v_real_moved) u
                 WHERE old_newest_tif = true
                 GROUP BY src_fid, src_tid) td
              JOIN mail.threads t
                ON (t.uid = i_uid AND tid = src_tid)) td,
          LATERAL (
            SELECT mid
              FROM mail.box
             WHERE uid = i_uid
               AND fid = td.src_fid
               AND tid = td.src_tid
             ORDER BY received_date DESC
             LIMIT 1) mm) mn_tif
    WHERE uid = i_uid
      AND mid = new_newest_mid;

END;
$$ LANGUAGE plpgsql;