CREATE OR REPLACE FUNCTION impl.update_threads(
    i_uid            bigint,
    current_revision bigint,
    moved            impl.moved_message[]
) RETURNS void AS $$
DECLARE
    c  record;
BEGIN
    /*
    add, delete or update mail.threads base on moved.{src_tid,dst_tid} changes
    */

    /* fill threads for messages with 'tid-reborn'
       (aka move from trash) */
    FOR c IN (
        SELECT t.tid, nt.dst_tid,
               moved_count, moved_seen,
               moved_attach_count, moved_attach_size,
               moved_labels,
               (newest_moved).mid AS moved_newest_mid,
               (newest_moved).received_date AS moved_newest_date
          FROM (
            SELECT dst_tid,
                   count(*) AS moved_count,
                   sum(seen::integer) AS moved_seen,
                   sum((ai).attach_count) AS moved_attach_count,
                   sum((ai).attach_size) AS moved_attach_size,
                   impl.sum(impl.make_thread_labels(lids,1)) AS moved_labels,
                   impl.newest_mid((mid, received_date)::impl.mid_with_date) AS newest_moved
              FROM unnest(moved) m
             WHERE m.src_tid IS NULL
               AND m.dst_tid IS NOT NULL
             GROUP BY dst_tid) nt
            LEFT JOIN mail.threads t
              ON (dst_tid = tid AND uid = i_uid) )
    LOOP
        IF c.tid IS NULL THEN -- fill thread for reborn
            INSERT INTO mail.threads
                (uid, tid, revision,
                newest_date, newest_mid,
                message_count, message_seen,
                attach_count, attach_size,
                labels)
            VALUES
                (i_uid, c.dst_tid, current_revision,
                c.moved_newest_date, c.moved_newest_mid,
                c.moved_count, c.moved_seen,
                c.moved_attach_count, c.moved_attach_size,
                c.moved_labels);
        ELSE
            UPDATE mail.threads t
               SET revision = current_revision,
                   newest_date = greatest(c.moved_newest_date, newest_date),
                   newest_mid = CASE WHEN c.moved_newest_date > newest_date
                                     THEN c.moved_newest_mid
                                     ELSE newest_mid END,
                   message_count = message_count + c.moved_count,
                   message_seen = message_seen + c.moved_seen,
                   attach_count = attach_count + c.moved_attach_count,
                   attach_size = attach_size + c.moved_attach_size,
                   labels = impl.sum_thread_labels(labels, c.moved_labels)
             WHERE t.uid = i_uid
               AND tid = c.dst_tid;
        END IF;
    END LOOP;

    -- if has dst_tid IS NULL (move to trash or remove_messages)
    DELETE FROM mail.threads t
     USING (
      SELECT src_tid, count(1) AS moved_count
        FROM unnest(moved)
       WHERE src_tid IS NOT NULL
         AND dst_tid IS NULL
       GROUP BY src_tid
      ) td
     WHERE uid = i_uid
       AND message_count = moved_count
       AND tid = src_tid;

    UPDATE mail.threads t
       SET revision = current_revision,
           message_count = message_count - moved_count,
           message_seen =  message_seen - moved_seen,
           attach_count = attach_count - moved_attach_count,
           attach_size = attach_size - moved_attach_size,
           labels = impl.minus_thread_labels(labels, labels_moved),
           newest_date = (result_newest).received_date,
           newest_mid = (result_newest).mid
      FROM (
        SELECT td.*,
               CASE WHEN ti.message_count = moved_count
                      OR (NOT ARRAY[ti.newest_mid] && moved_mids)
                    THEN (ti.newest_mid, ti.newest_date)::impl.mid_with_date
                    ELSE (
                      SELECT (mid, received_date)::impl.mid_with_date
                        FROM mail.box mb
                       WHERE uid = i_uid
                         AND mb.tid = td.src_tid
                       ORDER BY received_date DESC
                       LIMIT 1)
                    END AS result_newest
          FROM (
            SELECT src_tid,
                   count(*) AS moved_count,
                   sum(m.seen::integer) AS moved_seen,
                   sum((ai).attach_count) AS moved_attach_count,
                   sum((ai).attach_size) AS moved_attach_size,
                   impl.sum(impl.make_thread_labels(lids,1)) AS labels_moved,
                   array_agg(mid) AS moved_mids
              FROM unnest(moved) m
             WHERE dst_tid IS NULL
             GROUP BY src_tid ) td
          JOIN mail.threads ti
            ON (ti.uid = i_uid AND ti.tid = src_tid) ) td
     WHERE uid = i_uid
       AND tid = src_tid;

    -- set current_revision for unchanged threads
    UPDATE mail.threads t
       SET revision = current_revision
      FROM (
      SELECT src_tid AS unchanged_tid
        FROM unnest(moved)
       WHERE src_tid = dst_tid
       GROUP BY src_tid) tc
     WHERE uid = i_uid
       AND tid = unchanged_tid;
END;
$$ LANGUAGE plpgsql;