CREATE OR REPLACE FUNCTION impl.join_threads(
    i_uid     bigint,
    i_tid     bigint,
    join_tids bigint[],
    i_current_revision bigint
) RETURNS impl.short_changed_message[] AS $$
DECLARE
    check_thread     bigint;
    current_revision bigint;
    joined           impl.short_changed_message[];
BEGIN
    IF i_tid = ANY(join_tids) THEN
        RAISE EXCEPTION 'Can''t join threads % contains i_tid %', join_tids, i_tid
              USING HINT = 'Remove tid from join_tids';
    END IF;

    SELECT count(1)
      INTO check_thread
      FROM mail.threads
     WHERE uid = i_uid
       AND tid = i_tid;

    IF check_thread = 0 THEN
        RAISE EXCEPTION 'Can''t join threads % to nonexistent thread %', join_tids, i_tid
              USING TABLE = 'mail.threads';
    END IF;

    SELECT j_tid
      INTO check_thread
      FROM unnest(join_tids) j_tid
     WHERE NOT EXISTS (
        SELECT 1
          FROM mail.threads t
         WHERE uid = i_uid
           AND tid = j_tid
      );

    IF found THEN
        RAISE EXCEPTION 'Can''t join threads % to thread %,'
                        ' at least % tid not found', join_tids, i_tid, check_thread
              USING TABLE = 'mails.threads';
    END IF;

    WITH mails_joined AS (
        UPDATE mail.box
           SET tid = i_tid,
               newest_tif = false,
               newest_tit = impl.default_newest_tit(tab),
               revision = i_current_revision
         WHERE uid = i_uid
           AND tid = ANY(join_tids)
        RETURNING *)
    SELECT array_agg(
            (
              mid, tid, fid,
              seen, recent, deleted, lids, tab
            )::impl.short_changed_message)
      INTO joined
      FROM mails_joined;

    IF coalesce(cardinality(joined), 0) = 0 THEN
        RETURN joined;
    END IF;

    -- reset newest_tif @i_tid
    UPDATE mail.box
       SET newest_tif = false
     WHERE uid = i_uid
       AND tid = i_tid
       AND newest_tif = true;

    UPDATE mail.box m
       SET newest_tif = true
      FROM (
        SELECT fid AS join_fid,
               min(newest_joined_mid) AS newest_joined_mid
          FROM (
            SELECT fid,
                   last_value(mid) over (
                      PARTITION BY fid
                      ORDER BY received_date
                      ROWS
                        BETWEEN UNBOUNDED PRECEDING
                            AND UNBOUNDED FOLLOWING
                    ) AS newest_joined_mid
              FROM mail.box m
             WHERE m.uid = i_uid
               AND m.tid = i_tid) x
         GROUP BY fid) ntf
     WHERE uid = i_uid
       AND mid = newest_joined_mid;

    UPDATE mail.messages
       SET found_tid = i_tid
     WHERE uid = i_uid
       AND mid IN (
        SELECT mid
          FROM unnest(joined));

    -- reset newest_tit @i_tid
    UPDATE mail.box
       SET newest_tit = false
     WHERE uid = i_uid
       AND tid = i_tid
       AND newest_tit = true;

    UPDATE mail.box m
       SET newest_tit = true
      FROM (
        SELECT tab AS join_tab,
               min(newest_joined_mid) AS newest_joined_mid
          FROM (
            SELECT tab,
                   last_value(mid) over (
                      PARTITION BY tab
                      ORDER BY received_date
                      ROWS
                        BETWEEN UNBOUNDED PRECEDING
                            AND UNBOUNDED FOLLOWING
                    ) AS newest_joined_mid
              FROM mail.box m
             WHERE m.uid = i_uid
               AND tab IS NOT NULL
               AND m.tid = i_tid) x
         GROUP BY tab) ntt
     WHERE uid = i_uid
       AND mid = newest_joined_mid;

    UPDATE mail.messages
       SET found_tid = i_tid
     WHERE uid = i_uid
       AND mid IN (
        SELECT mid
          FROM unnest(joined));

    UPDATE mail.threads_hashes
       SET tid = i_tid
     WHERE uid = i_uid
       AND tid = ANY(join_tids);

    WITH joined_threads AS (
        DELETE FROM mail.threads
         WHERE uid = i_uid
           AND tid = ANY(join_tids)
        RETURNING *
    ), joined_threads_and_into AS (
        SELECT t.*
          FROM mail.threads t
         WHERE t.uid = i_uid
           AND t.tid = i_tid
         UNION ALL
        SELECT ot.*
          FROM joined_threads ot
    )
    UPDATE mail.threads t
       SET revision = i_current_revision,
           message_count = r.message_count,
           message_seen = r.message_seen,
           attach_count = r.attach_count,
           attach_size = r.attach_size,
           newest_date = r.newest_date,
           newest_mid = r.newest_mid,
           labels = l.labels
      FROM
        (SELECT sum(message_count) AS message_count,
                sum(message_seen) AS message_seen,
                sum(attach_count) AS attach_count,
                sum(attach_size) AS attach_size,
                max(newest_mid) AS newest_mid,
                max(newest_date) AS newest_date
          FROM (
            SELECT
                  message_count, message_seen,
                  attach_count, attach_size,
                  max(newest_date) OVER (ORDER BY newest_date DESC) AS newest_date,
                  max(newest_mid) OVER (ORDER BY newest_date DESC) AS newest_mid
              FROM joined_threads_and_into ) t ) r,
       (SELECT array_agg((lid, message_count)::mail.thread_label) as labels
          FROM (
            SELECT lid, sum(message_count) AS message_count
              FROM (
                SELECT l[i].lid AS lid, l[i].message_count AS message_count
                  FROM (
                    SELECT labels AS l, generate_subscripts(labels, 1) AS i
                      FROM joined_threads_and_into tj) lt
                    ) lti GROUP BY lid
                ) ls
            ) l
       WHERE t.uid = i_uid
         AND t.tid = i_tid;

    UPDATE mail.folders
       SET revision = i_current_revision
     WHERE uid = i_uid
       AND fid IN (
        SELECT fid
          FROM unnest(joined)
        );

    UPDATE mail.labels
       SET revision = i_current_revision
     WHERE uid = i_uid
       AND lid IN (
        SELECT unnest(lids) lid
          FROM unnest(joined)
        );

    UPDATE mail.tabs
       SET revision = i_current_revision
     WHERE uid = i_uid
       AND tab IN (
        SELECT tab
          FROM unnest(joined)
         WHERE tab IS NOT NULL
        );

    RETURN joined;
END;
$$ LANGUAGE plpgsql;
