CREATE OR REPLACE FUNCTION impl.store_message(
    i_uid        bigint,
    i_coords     impl.store_coordinates,
    i_headers    impl.store_headers,
    i_recipients mail.recipient[],
    i_attaches   mail.attach[],
    i_mime       mail.mime_part[],
    i_lids       integer[],
    i_threads    impl.store_threading
) RETURNS impl.store_result AS $$
DECLARE
    -- out
    v_mid             bigint;
    v_tid             bigint;
    v_imap_id         bigint;
    current_revision  bigint;
    v_fresh_count     bigint;

    -- local variables
    v_mid_serial  bigint;
    v_folder_type mail.folder_types;

    v_attach_size    integer;
    v_attach_count   integer;
    v_seen_increment integer := i_coords.seen::integer;

    -- threads
    v_need_thread        boolean;
    -- tif thread-in-folder
    v_newest_tif         boolean;
    -- tit thread-in-tab
    v_newest_tit             boolean;
    v_prev_newest_mid    bigint;
    v_update_prev_newest boolean;

    -- del chains
    v_prev_chain           smallint;
    v_prev_chained_imap_id bigint;
    v_start_new_chain      boolean;

    v_pop3initialized boolean;
BEGIN
    IF i_coords.tab IS NOT NULL AND NOT EXISTS (
        SELECT 1
          FROM mail.tabs t
         WHERE t.uid = i_uid
           AND t.tab = i_coords.tab
    ) THEN
        RAISE EXCEPTION 'Can''t store in tab % for uid %, it does not exist', i_coords.tab, i_uid;
    END IF;

    -- replace this logic with domain
    SELECT coalesce(sum((a).size), 0),
           count(*)
      INTO v_attach_size, v_attach_count
      FROM unnest(i_attaches) a;

    UPDATE mail.serials
       SET next_revision = next_revision + 1,
           next_mid_serial = next_mid_serial + 1
     WHERE uid = i_uid
    RETURNING next_revision - 1, next_mid_serial - 1
      INTO current_revision, v_mid_serial;

    PERFORM impl.assert_user_is_here(i_uid);

    PERFORM impl.check_uid_lids(i_uid, i_lids);

    v_mid := impl.make_mid(i_coords.received_date, v_mid_serial);
    -- treat tid IS NULL as new thread
    v_tid := coalesce(i_coords.tid, v_mid);

    INSERT INTO mail.messages
        (uid, mid,
        st_id, size,
        attaches,
        mime,
        attributes,
        subject, firstline,
        hdr_date, hdr_message_id,
        recipients,
        extra_data,
        found_tid,
        thread_rule)
    VALUES
        (i_uid, v_mid,
        i_coords.st_id, i_coords.size,
        -- emulate original store_messages
        -- store empty array as NULL
        nullif(i_attaches, ARRAY[]::mail.attach[]),
        i_mime,
        i_coords.attributes,
        i_headers.subject, i_headers.firstline,
        i_headers.hdr_date, i_headers.hdr_message_id,
        i_recipients,
        i_headers.extra_data,
        v_tid,
        i_threads.rule);

    INSERT INTO mail.message_references
        (uid, mid, type, value)
    SELECT i_uid, v_mid, 'reference'::mail.message_reference_type, ref_value
      FROM unnest(i_threads.references_hashes) ref_value
     WHERE (i_threads.in_reply_to_hash IS NULL OR ref_value != i_threads.in_reply_to_hash)
     UNION ALL
    SELECT i_uid, v_mid, 'in-reply-to'::mail.message_reference_type, i_threads.in_reply_to_hash
     WHERE i_threads.in_reply_to_hash IS NOT NULL;


    UPDATE mail.folders
       SET message_count = message_count + 1,
           message_seen = message_seen + v_seen_increment,
           message_recent = message_recent + 1,
           attach_count = attach_count + v_attach_count,
           attach_size = attach_size + v_attach_size,
           message_size = message_size + i_coords.size,
           revision = current_revision,
           next_imap_id = next_imap_id + 1,
           first_unseen =    CASE WHEN first_unseen_id IS NULL AND NOT i_coords.seen
                                  THEN message_count + 1
                                  ELSE first_unseen
                              END,
           first_unseen_id = CASE WHEN first_unseen_id IS NULL AND NOT i_coords.seen
                                  THEN next_imap_id
                                  ELSE first_unseen_id
                              END,
           unvisited =       CASE WHEN type IN ('inbox', 'user')
                                  THEN NOT i_coords.seen
                                  ELSE unvisited
                              END
     WHERE uid = i_uid
       AND fid = i_coords.fid
    RETURNING next_imap_id - 1,
              type,
              impl.is_threaded_folder(type),
              (pop3state).initialized
      INTO v_imap_id,
           v_folder_type,
           v_need_thread,
           v_pop3initialized;

    IF NOT found THEN
        RAISE EXCEPTION 'Nonexistent folder fid: % uid: %', i_coords.fid, i_uid
              USING TABLE = 'mail.folders';
    END IF;

    UPDATE mail.tabs
       SET message_count = message_count + 1,
           message_seen = message_seen + v_seen_increment,
           attach_count = attach_count + v_attach_count,
           attach_size = attach_size + v_attach_size,
           message_size = message_size + i_coords.size,
           revision = current_revision,
           unvisited = NOT i_coords.seen,
           fresh_count = fresh_count + (NOT i_coords.seen)::integer
     WHERE uid = i_uid
       AND tab = i_coords.tab;

    SELECT imap_id, chain
      INTO v_prev_chained_imap_id, v_prev_chain
      FROM mail.box
     WHERE uid = i_uid
       AND fid = i_coords.fid
       AND chain IS NOT NULL
     ORDER BY imap_id DESC
     LIMIT 1;

    v_start_new_chain := v_prev_chain IS NULL OR v_prev_chain >= impl.get_chain_size();

    IF v_need_thread THEN
        -- newest_TIF (Thread-In-Folder)
        -- There are 3 cases:
        -- 1. If there are no our thread in our folder
        --    then our messages is `newest_tif`
        -- 2. If there is newest_tif message
        --    for our-thread-in-our-folder and
        --    it's received_date <= our receive_date, then:
        --      * our messages is `newest_tif`
        --      * we should remove `newest_tif` from previous messages
        -- 3. If there is newest_tif message for our-thread-in-our-folder, but
        --    it's received_date > then our received_date,
        --    then our message is not `newest_tif`
        v_newest_tif := true;
        IF EXISTS (
            SELECT 1
              FROM mail.threads t
             WHERE t.uid = i_uid
               AND t.tid = v_tid
        ) THEN
            SELECT mid, received_date <= i_coords.received_date
              INTO v_prev_newest_mid, v_update_prev_newest
              FROM mail.box
             WHERE uid = i_uid
               AND fid = i_coords.fid
               AND tid = v_tid
               AND newest_tif;

            IF found THEN
                -- there is newest_tif message for our-thread-in-our-folder
                IF v_update_prev_newest THEN
                    -- Case #2.
                    UPDATE mail.box
                       SET newest_tif = false
                     WHERE uid = i_uid
                       AND mid = v_prev_newest_mid;
                ELSE
                    -- Case #3.
                    v_newest_tif := false;
                END IF;
            END IF;
        END IF;
    ELSE
        -- Don't need neweset_tif pointers
        -- for non threaded folders
        v_newest_tif := false;
    END IF;

    IF i_coords.tab IS NOT NULL THEN
        -- newest_tit (Thread-In-Tab)
        -- There are 3 cases:
        -- 1. If there are no our thread in our tab
        --    then our messages is `newest_tif`
        -- 2. If there is newest_tit message
        --    for our-thread-in-our-tab and
        --    it's received_date <= our receive_date, then:
        --      * our messages is `newest_tit`
        --      * we should remove `newest_tit` from previous messages
        -- 3. If there is newest_tit message for our-thread-in-our-tab, but
        --    it's received_date > then our received_date,
        --    then our message is not `newest_tif`
        v_newest_tit := true;
        IF EXISTS (
            SELECT 1
              FROM mail.threads t
             WHERE t.uid = i_uid
               AND t.tid = v_tid
        ) THEN
            SELECT mid, received_date <= i_coords.received_date
              INTO v_prev_newest_mid, v_update_prev_newest
              FROM mail.box
             WHERE uid = i_uid
               AND tab = i_coords.tab
               AND tid = v_tid
               AND newest_tit;

            IF found THEN
                -- there is newest_tit message for our-thread-in-our-tab
                IF v_update_prev_newest THEN
                    -- Case #2.
                    UPDATE mail.box
                       SET newest_tit = false
                     WHERE uid = i_uid
                       AND mid = v_prev_newest_mid;
                ELSE
                    -- Case #3.
                    v_newest_tit := false;
                END IF;
            END IF;
        END IF;
    ELSE
        -- Should set null newest_tit
        -- for non tabbed messages
        v_newest_tit := NULL;
    END IF;

    INSERT INTO mail.box
        (uid, mid,
        fid,
        tid,
        newest_tif,
        imap_id, revision,
        chain,
        seen, recent, deleted,
        received_date,
        lids,
        doom_date,
        tab, newest_tit)
    VALUES
        (i_uid, v_mid,
        i_coords.fid,
        CASE WHEN v_need_thread THEN v_tid ELSE NULL END,
        v_newest_tif,
        v_imap_id, current_revision,
        CASE WHEN v_start_new_chain THEN 1 ELSE NULL END,
        i_coords.seen, true, i_coords.deleted,
        i_coords.received_date,
        i_lids,
        CASE WHEN impl.is_doom_folder(v_folder_type) THEN now() ELSE NULL END,
        i_coords.tab, v_newest_tit);

    IF NOT v_start_new_chain THEN
        UPDATE mail.box
           SET chain = chain + 1
         WHERE uid = i_uid
           AND fid = i_coords.fid
           AND imap_id = v_prev_chained_imap_id;
    END IF;

    IF v_need_thread THEN
        INSERT INTO mail.threads
            (uid, tid, revision,
            newest_date, newest_mid,
            message_count, message_seen,
            attach_count, attach_size,
            labels)
        VALUES
            (i_uid, v_tid, current_revision,
            i_coords.received_date, v_mid,
            1, v_seen_increment,
            v_attach_count, v_attach_size,
            impl.make_thread_labels(i_lids, 1))
        ON CONFLICT (uid, tid) DO UPDATE
           SET revision = current_revision,
               message_count = threads.message_count + 1,
               message_seen = threads.message_seen + v_seen_increment,
               newest_date = greatest(i_coords.received_date, threads.newest_date),
               newest_mid = CASE WHEN threads.newest_date > i_coords.received_date
                                 THEN threads.newest_mid
                                 ELSE v_mid END,
               attach_count = threads.attach_count + v_attach_count,
               attach_size = threads.attach_size + v_attach_size,
               labels = impl.inc_thread_labels(threads.labels, i_lids);

        IF i_threads.rule = 'hash' THEN
            INSERT INTO mail.threads_hashes
                (uid, namespace,
                value, uniq_key,
                tid)
            VALUES
                (i_uid, i_threads.hash_namespace,
                i_threads.hash_value, i_threads.hash_uniq_key,
                v_tid)
            ON CONFLICT (uid, namespace, value, uniq_key)
            DO UPDATE SET tid = v_tid;
        END IF;
    END IF;

    IF v_pop3initialized THEN
        INSERT INTO mail.pop3_box
            (uid, mid, fid, size)
        VALUES
            (i_uid, v_mid, i_coords.fid, i_coords.size);
    END IF;

    IF v_need_thread AND array_length(i_lids, 1) > 0 THEN
        UPDATE mail.labels
           SET revision = current_revision,
               message_count = message_count + 1,
               message_seen = message_seen + v_seen_increment
         WHERE uid = i_uid
           AND lid = ANY(i_lids);
    END IF;

    IF NOT impl.is_hidden_folder(v_folder_type) AND v_attach_count > 0 THEN
        UPDATE mail.counters
           SET has_attaches_count = has_attaches_count + 1,
               has_attaches_seen = has_attaches_seen + v_seen_increment,
               revision = current_revision
         WHERE uid = i_uid;
    END IF;

    IF impl.fresh_predicate(v_folder_type) AND NOT i_coords.seen THEN
        UPDATE mail.counters
           SET fresh_count = fresh_count + 1,
               revision = current_revision
         WHERE uid = i_uid
        RETURNING fresh_count INTO v_fresh_count;
    ELSE
        SELECT fresh_count
          INTO v_fresh_count
          FROM mail.counters
         WHERE uid = i_uid;
    END IF;

    RETURN (
        v_mid,
        v_tid,
        i_coords.fid,
        v_imap_id,
        i_coords.seen,
        true, -- recent
        i_coords.deleted,
        i_lids,
        i_headers.hdr_message_id,
        v_fresh_count,
        current_revision,
        i_coords.tab
    )::impl.store_result;
END;
$$ LANGUAGE plpgsql;
