CREATE OR REPLACE FUNCTION impl.move_messages_impl(
    i_uid               bigint,
    i_mids              bigint[],
    i_dst_fid           integer,
    i_dst_tab           mail.tab_types,
    i_current_revision  bigint
) RETURNS impl.moved_message[] AS $$
DECLARE
    hide_threads         boolean;
    doom_folder          boolean;
    folder_next_imap_id  bigint;
    last_chained         impl.chained_message;
    moved                impl.moved_message[];
BEGIN
    /*
      1. update mail.box and define what really @moved
      2. update old and new folders
      3. update old and new tabs
      4. update IMAP chains
      5. fill new threads
      6. update old and new threads_folders
      7. update old threads
      8. labels
    */

    /* get destination folder info - this will affect final values in mail.box
    */
    SELECT NOT impl.is_threaded_folder(type), next_imap_id, impl.is_doom_folder(type)
      INTO hide_threads, folder_next_imap_id, doom_folder
      FROM mail.folders
     WHERE uid = i_uid
       AND fid = i_dst_fid;

    /* check if destination folder exists
    */
    IF NOT found THEN
        RAISE EXCEPTION 'Nonexistent folder fid: %, uid: %', i_dst_fid, i_uid
              USING HINT = 'Please check your user ID',
                    TABLE = 'mail.folders';
    END IF;

    /* check if destination tab exists
    */
    IF i_dst_tab IS NOT NULL AND NOT EXISTS (
        SELECT 1
          FROM mail.tabs t
         WHERE t.uid = i_uid
           AND t.tab = i_dst_tab
    ) THEN
        RAISE EXCEPTION 'Can''t move to tab % for uid %, it does not exist', i_dst_tab, i_uid;
    END IF;

    last_chained := impl.get_last_chained(i_uid, i_dst_fid);

    WITH with_moved AS (
        UPDATE mail.box m
               /* there are 3 types of fields:
                  1. That should be changed only when folder was changed.
                  2. That should be changed only when tab was changed.
                  3. That should be changed regardless.

                  1 & 2 are changed via CASE WHEN flag THEN new_val ELSE old_val END
                        (checking the flags we got in select below)
                  3 are changed as usual
               */
           SET revision = i_current_revision,
               fid = CASE WHEN ml.change_fid
                          THEN i_dst_fid
                          ELSE fid END,
               newest_tif = CASE WHEN ml.change_fid
                                 THEN false
                                 ELSE newest_tif END,
               recent = CASE WHEN ml.change_fid
                             THEN true
                             ELSE recent END,
               tid = CASE WHEN ml.change_fid
                          THEN CASE WHEN hide_threads
                                    THEN NULL
                                    WHEN tid IS NULL
                                    THEN impl.get_found_tid(i_uid, m.mid)
                                    ELSE tid END
                          ELSE tid END,
               chain = CASE WHEN ml.change_fid
                            THEN impl.new_chain_formula(moved_between_folders_number, moved_between_folders_count, last_chained)
                            ELSE chain END,
               imap_id = CASE WHEN ml.change_fid
                              THEN (folder_next_imap_id + moved_between_folders_number - 1)
                              ELSE imap_id END,
               doom_date = CASE WHEN ml.change_fid
                                THEN CASE WHEN doom_folder
                                          THEN now()
                                          ELSE NULL END
                                ELSE doom_date END,
               tab = CASE WHEN ml.change_tab
                          THEN i_dst_tab
                          ELSE tab END,
               newest_tit = CASE WHEN ml.change_tab
                                 THEN impl.default_newest_tit(i_dst_tab)
                                 ELSE newest_tit END
          FROM (
            SELECT mid,
                   /* get row number an total count for messages that will be moved between folders
                      this will be used for imap counters calculation
                   */
                   sum((fid != i_dst_fid)::integer) OVER (ORDER BY mid) AS moved_between_folders_number,
                   sum((fid != i_dst_fid)::integer) OVER () AS moved_between_folders_count,

                   /* get original values of what we will change
                   */
                   fid AS src_fid,
                   recent AS old_recent,
                   tid AS src_tid,
                   chain AS src_chain,
                   imap_id AS src_imap_id,
                   newest_tif AS old_newest_tif,
                   tab AS src_tab,
                   newest_tit AS old_newest_tit,

                   /* get flags if folder or tab will be changed
                   */
                   (fid != i_dst_fid) AS change_fid,
                   (tab IS DISTINCT FROM i_dst_tab) AS change_tab
              FROM mail.box
             WHERE uid = i_uid
                   /* we should update only messages that are not in destination folder or tab
                   */
               AND (fid != i_dst_fid OR tab IS DISTINCT FROM i_dst_tab)
               AND mid = ANY(i_mids)
          ) ml
         WHERE m.uid = i_uid
           AND m.mid = ml.mid
        RETURNING m.mid, ml.src_fid,
                  m.tid, ml.src_tid,
                  m.imap_id, ml.src_imap_id,
                  ml.src_chain,
                  m.seen, ml.old_recent, m.deleted,
                  m.lids, m.received_date, old_newest_tif,
                  ml.src_tab, old_newest_tit)
    SELECT array_agg((
        um.mid, i_dst_fid, um.src_fid,
        um.tid, um.src_tid,
        um.imap_id, um.src_imap_id,
        um.src_chain,
        um.seen, um.old_recent, deleted,
        um.lids,
        mm.size,
        um.received_date,
        old_newest_tif,
        impl.get_attaches_info(mm.attaches),
        hdr_message_id,
        mm.attributes,
        i_dst_tab, um.src_tab, old_newest_tit
        )::impl.moved_message)
      INTO moved
      FROM with_moved um
      JOIN mail.messages mm
        ON (um.mid = mm.mid and mm.uid = i_uid);

    IF moved IS NULL OR cardinality(moved) = 0 THEN
        RETURN moved;
    END IF;

    /* update a bunch of counters after move
    */
    PERFORM impl.update_src_tabs(
        i_uid,
        i_current_revision,
        moved
    );

    PERFORM impl.update_src_folders(
        i_uid,
        i_current_revision,
        moved
    );

    PERFORM impl.update_threads(
        i_uid,
        i_current_revision,
        moved
    );

    PERFORM impl.update_dst_tab(
        i_uid,
        i_dst_tab,
        i_current_revision,
        moved
    );

    PERFORM impl.update_dst_folder(
        i_uid,
        i_dst_fid,
        i_current_revision,
        moved,
        last_chained
    );

    PERFORM impl.update_labels(
        i_uid,
        i_current_revision,
        moved
    );

    PERFORM impl.update_pop3(
        i_uid,
        moved
    );

    PERFORM impl.update_counters(
        i_uid,
        i_current_revision,
        moved
    );

    RETURN moved;
END;
$$ LANGUAGE plpgsql;
