CREATE OR REPLACE FUNCTION impl.update_messages_impl(
    i_uid              bigint,
    i_mids             bigint[],
    set_seen           boolean,
    set_recent         boolean,
    set_deleted        boolean,
    i_lids_add         integer[],
    i_lids_del         integer[],
    i_current_revision bigint
) RETURNS impl.updated_message[] AS $$
DECLARE
    updated          impl.updated_message[];
BEGIN
    IF coalesce(set_seen, set_recent, set_deleted) IS NULL AND #i_lids_add = 0 AND #i_lids_del = 0 THEN
        RAISE EXCEPTION 'Invalid arguments, set_seen: %, set_recent: %, set_deleted: %,'
                         ' i_lids_add: % and i_lids_del: %', set_seen, set_recent, set_deleted,
                         i_lids_add, i_lids_del
                  USING HINT = 'Verify arguments';
    END IF;

    IF i_lids_add && i_lids_del THEN
        RAISE EXCEPTION 'Got i_lids_add: % and i_lids_del: % intersection', i_lids_add, i_lids_del;
    END IF;

    PERFORM impl.check_uid_lids(i_uid, i_lids_add);
    PERFORM impl.check_uid_lids(i_uid, i_lids_del);

    WITH with_updated AS (
        UPDATE mail.box m
           SET seen    = coalesce(set_seen, seen),
               recent  = coalesce(set_recent, recent),
               deleted = coalesce(set_deleted, deleted),
               lids = uniq(sort(coalesce(lids, ARRAY[]::integer[]) - i_lids_del + i_lids_add)),
               revision = i_current_revision
         WHERE uid = i_uid
           AND mid = ANY(i_mids)
           AND ((set_seen    IS NOT NULL AND seen    != set_seen) OR
                (set_recent  IS NOT NULL AND recent  != set_recent) OR
                (set_deleted IS NOT NULL AND deleted != set_deleted) OR
                (#i_lids_add > 0 AND (lids IS NULL OR (NOT (lids @> i_lids_add)))) OR
                (#i_lids_del > 0 AND (lids IS NOT NULL AND lids && i_lids_del)))
        RETURNING *)
    SELECT array_agg((
        om.mid, om.fid, om.tid, om.imap_id,
        um.revision,
        um.seen::integer    - om.seen::integer,
        um.recent::integer  - om.recent::integer,
        um.deleted::integer - om.deleted::integer,
        impl.make_thread_labels(um.lids - om.lids, 1) || impl.make_thread_labels(om.lids - um.lids, -1),
        impl.make_thread_labels(um.lids - om.lids, um.seen::integer)
          || impl.make_thread_labels(um.lids & om.lids, um.seen::integer - om.seen::integer)
          || impl.make_thread_labels(om.lids - um.lids, -1 * om.seen::integer),
        um.seen, um.recent, um.deleted,
        um.lids, om.tab
        )::impl.updated_message)
      INTO updated
      FROM with_updated um
      JOIN mail.box om
        ON om.mid = um.mid
     WHERE om.uid = i_uid
       AND om.mid = ANY(i_mids);

    IF updated IS NULL OR cardinality(updated) = 0 THEN
        RETURN updated;
    END IF;

    UPDATE mail.folders f
       SET message_seen    = message_seen + d.seen_change,
           message_recent  = message_recent + d.recent_change,
           revision        = i_current_revision,
           first_unseen    = (fu_def).first_unseen,
           first_unseen_id = (fu_def).first_unseen_id
      FROM (
        SELECT d.*,
               CASE WHEN d.seen_change > 0
                              -- we mark all message as seen - no unseen in current folder
                    THEN CASE WHEN (d.seen_change + message_seen) = message_count
                              THEN (0, null)::impl.first_unseen_def
                              -- there is unseen message in folder with smaller imap_id
                              WHEN fi.first_unseen_id < min_imap_id
                              THEN (fi.first_unseen, fi.first_unseen_id)::impl.first_unseen_def
                              -- we mark previous first_unseen as seen
                              -- and exist unseen message with greater imap_id
                              ELSE impl.find_first_unseen_by_pos(
                                      i_uid => i_uid,
                                      i_fid => d.fid,
                                      i_imap_id => impl.find_first_unseen_id_greater_then_id(
                                          i_uid => i_uid,
                                          i_fid => d.fid,
                                          i_imap_id => fi.first_unseen_id
                                      ))
                         END
                    WHEN d.seen_change < 0
                              -- we mark as unseen message with smaller imap_id
                              -- use it as first unseen
                    THEN CASE WHEN fi.first_unseen_id IS NULL OR fi.first_unseen_id > min_imap_id
                              THEN impl.find_first_unseen_by_pos(i_uid, d.fid, min_imap_id)
                              -- mark as unseen message with greater imap_id
                              -- don't touch previous first unseen
                              ELSE (fi.first_unseen, fi.first_unseen_id)::impl.first_unseen_def
                         END
                    ELSE
                        (fi.first_unseen, fi.first_unseen_id)::impl.first_unseen_def
                END AS fu_def
          FROM mail.folders fi, (
            SELECT (u).fid AS fid,
                   sum((u).seen_change) AS seen_change,
                   sum((u).recent_change) AS recent_change,
                   min((u).imap_id) AS min_imap_id
              FROM unnest(updated) u
             GROUP BY (u).fid) d
         WHERE fi.uid = i_uid
           AND fi.fid = d.fid) d
    WHERE uid = i_uid
      AND f.fid = d.fid;

    UPDATE mail.tabs t
       SET message_seen    = message_seen + d.seen_change,
           revision        = i_current_revision
      FROM (
        SELECT d.*
          FROM mail.tabs ti, (
            SELECT (u).tab AS tab,
                   sum((u).seen_change) AS seen_change
              FROM unnest(updated) u
             WHERE tab IS NOT NULL
             GROUP BY (u).tab) d
         WHERE ti.uid = i_uid
           AND ti.tab = d.tab) d
    WHERE uid = i_uid
      AND t.tab = d.tab;

    UPDATE mail.threads t
       SET message_seen = message_seen + dt.seen_change,
           revision = i_current_revision,
           labels = impl.sum_thread_labels(t.labels, dt.lids_diff)
      FROM (
       SELECT (u).tid AS tid,
              sum((u).seen_change) AS seen_change,
              impl.sum((u).lids_diff) AS lids_diff
         FROM unnest(updated) u
        WHERE u.tid IS NOT NULL
        GROUP BY (u).tid
        ) dt
     WHERE t.uid = i_uid
       AND t.tid = dt.tid;

    IF #i_lids_add > 0 OR #i_lids_del > 0 OR set_seen IS NOT NULL THEN
        UPDATE mail.labels ld
           SET revision = i_current_revision,
               message_seen  = ld.message_seen  + COALESCE(dt.message_seen, 0),
               message_count = ld.message_count + COALESCE(dt.message_count, 0)
          FROM (
            SELECT coalesce((ll).lid, (ls.lid)) AS lid,
                   (ll).message_count AS message_count,
                   (ls).message_count AS message_seen
              FROM (
                SELECT impl.sum((u).lids_diff) AS lids_diff,
                       impl.sum((u).lids_seen_diff) AS lids_seen_diff
                  FROM unnest(updated) u
                 WHERE (u).tid IS NOT NULL) xl,
                  unnest(xl.lids_seen_diff) ls FULL OUTER JOIN unnest(xl.lids_diff) ll ON ll.lid = ls.lid ) dt
         WHERE ld.uid = i_uid
           AND (dt.message_count != 0 OR dt.message_seen != 0)
           AND dt.lid = ld.lid;
    END IF;

    UPDATE mail.counters c
       SET has_attaches_seen = has_attaches_seen + d.seen_change,
           revision = i_current_revision
      FROM (
        SELECT coalesce(sum((u).seen_change), 0) AS seen_change
          FROM unnest(updated) u
          JOIN mail.messages m
            ON (m.uid = i_uid AND m.mid = u.mid)
         WHERE cardinality(m.attaches) > 0
      ) d
     WHERE c.uid = i_uid;

    RETURN updated;
END;
$$ LANGUAGE plpgsql;
