#!/usr/bin/env python
# coding: utf-8

from __future__ import print_function
from contextlib import closing
import os
import subprocess
import yaml

MDB_COMPONENT = 'components/pg-code/maildb'

def logged(func):
    def wrap(*args, **kwargs):
        res = func(*args, **kwargs)
        print('%s(%r, %r) => %r' % (func.__name__, args, kwargs, res))
        return res
    return wrap

@logged
def command(*args):
    cmd = subprocess.Popen(args)
    cmd.communicate()
    assert not cmd.returncode, \
        '%r - %r exit with %r' % (args, cmd, cmd.returncode)

@logged
def find_tags(filename):
    with closing(open(filename)) as fd:
        all_versions = yaml.load(fd)
        versions = {}
        for tag_name, all_components in all_versions.iteritems():
            mdb_version = all_components.get(MDB_COMPONENT)
            if mdb_version is not None:
                versions[tag_name] = mdb_version
        return versions


@logged
def clone_mdb(mdb_repo, mdb_dir):
    command(
        'git', 'clone', mdb_repo, mdb_dir
    )

@logged
def apply_tags(version):
    for tag_name, commit in version.iteritems():
        command(
            'git', 'tag', '-f', tag_name, commit
        )

@logged
def push_tags():
    command(
        'git', 'push', 'origin', '--tags', '-f'
    )

VERSIONS_FILE = 'pillar/versions.sls'
MDB_REPO = 'git@github.yandex-team.ru:mail/mdb.git'
MDB_DIR = 'MdB_DiR'

def main():
    tags = find_tags(VERSIONS_FILE)
    clone_mdb(MDB_REPO, MDB_DIR)
    os.chdir(MDB_DIR)
    apply_tags(tags)
    push_tags()

if __name__ == '__main__':
    main()
