DROP SCHEMA IF EXISTS mail CASCADE;
DROP SCHEMA IF EXISTS mailish CASCADE;

CREATE SCHEMA mail;

CREATE TYPE mail.user_state AS ENUM (
    'active',
    'inactive',
    'notified',
    'frozen',
    'archived',
    'deleted',
    'special'
);

-- XENO uses current logic of transfer/purging users and lays on timestamps
-- Do not change it without notifying XENO team
CREATE TABLE mail.users (
    uid        bigint NOT NULL,
    here_since timestamp with time zone NOT NULL DEFAULT current_timestamp,
    is_here    boolean NOT NULL DEFAULT true,
    data_version integer NOT NULL DEFAULT constants.newest_data_version(),
    is_deleted      boolean NOT NULL DEFAULT false,
    purge_date      timestamp with time zone,
    can_read_tabs   boolean DEFAULT true,
    state               mail.user_state NOT NULL DEFAULT 'active',
    last_state_update   timestamp with time zone NOT NULL DEFAULT current_timestamp,
    notifies_count      smallint NOT NULL DEFAULT 0,

    CONSTRAINT pk_users PRIMARY KEY (uid),
    CONSTRAINT check_data_version CHECK (
        data_version BETWEEN
            constants.minimal_data_version() AND
            constants.newest_data_version()
    )
);

CREATE TABLE mail.serials(
    uid                  bigint  NOT NULL,
    next_revision        bigint  NOT NULL,
    next_fid             integer NOT NULL,
    next_lid             integer NOT NULL,
    next_mid_serial      bigint  NOT NULL,
    next_owner_subscription_id      bigint  NOT NULL,
    next_subscriber_subscription_id bigint  NOT NULL,
    next_collector_id    integer,
    next_backup_id       integer NOT NULL,

    CONSTRAINT pk_serials PRIMARY KEY (uid),
    CONSTRAINT fk_serials_uid_users FOREIGN KEY (uid)
        REFERENCES mail.users ON DELETE CASCADE,
    CONSTRAINT next_collector_id_not_null
        CHECK (next_collector_id IS NOT NULL)
);

ALTER TABLE mail.users
    ADD CONSTRAINT fk_users_uid_serials
    FOREIGN KEY (uid) REFERENCES mail.serials
    DEFERRABLE INITIALLY DEFERRED;

CREATE TYPE mail.folder_types AS ENUM (
    'inbox',
    'spam',
    'trash',
    'sent',
    'outbox',
    'drafts',
    'archive',
    'templates',
    'discount',
    'user',
    'zombie',
    'unsubscribe',
    'pending',
    'hidden_trash',
    'restored', 
    'reply_later'
);

CREATE TYPE mail.pop3state AS (
    initialized boolean, -- Indicates POP3 was enabled once, and we need to synchronize with pop3 data
    enabled     boolean  -- Indicates POP3 is enabled for the folder
);


CREATE TABLE mail.folders (
    uid            bigint  NOT NULL,
    fid            integer NOT NULL,
    revision       bigint  NOT NULL,
    name           text    NOT NULL,
    parent_fid     integer,
    type           mail.folder_types NOT NULL,
    unvisited      boolean   NOT NULL DEFAULT 'no',
    unique_type    boolean   NOT NULL DEFAULT 'no',
    created        timestamp with time zone NOT NULL DEFAULT current_timestamp,
    -- IMAP specific meta
    next_imap_id    bigint  NOT NULL DEFAULT 1,
    uidvalidity     bigint  NOT NULL,
    first_unseen    integer NOT NULL DEFAULT 0,
    first_unseen_id bigint,

    message_count  integer NOT NULL DEFAULT 0,
    message_seen   integer NOT NULL DEFAULT 0,
    message_size   bigint  NOT NULL DEFAULT 0,
    message_recent integer NOT NULL DEFAULT 0, -- IMAP
    attach_count   integer NOT NULL DEFAULT 0,
    attach_size    bigint  NOT NULL DEFAULT 0,
    pop3state      mail.pop3state NOT NULL DEFAULT (false, false), -- POP3
    position       integer NOT NULL DEFAULT 0,

    subscribed_for_shared_folder     boolean,
    CONSTRAINT check_message_count CHECK (
        message_count >= 0 AND
        message_seen  >= 0 AND
        message_count >= message_seen
    ),
    CONSTRAINT check_message_recent CHECK (
        message_recent >= 0 AND
        message_count >= message_recent
    ),
    CONSTRAINT check_message_size CHECK (
        (message_size > 0 AND message_count > 0)
        OR
        (message_size = 0 AND message_count = 0)
    ),
    CONSTRAINT check_attaches CHECK (
        attach_count >= 0 AND
        attach_size >= 0 AND
        attach_size <= message_size
    ),
    CONSTRAINT check_next_imap_id CHECK (
        next_imap_id > 0 AND
        next_imap_id > message_count
    ),
    CONSTRAINT check_first_unseen CHECK (
        (first_unseen = 0 AND first_unseen_id IS NULL) -- all seen or no messages
        OR
        (first_unseen > 0 AND
         first_unseen <= message_count AND
         first_unseen_id IS NOT NULL AND
         first_unseen_id < next_imap_id)
    ),
    CONSTRAINT check_revision CHECK (
        revision > 0
    ),
    CONSTRAINT check_no_self_cycles CHECK (
        parent_fid IS NULL OR
        fid != parent_fid
    ),
    CONSTRAINT check_non_empty_name CHECK (
        name != ''
    ),
    CONSTRAINT check_fid_natural CHECK (fid > 0),
    CONSTRAINT pk_folders PRIMARY KEY (uid, fid),
    CONSTRAINT fk_folders_uid_parent_fid FOREIGN KEY (uid, parent_fid)
        REFERENCES mail.folders ON DELETE RESTRICT DEFERRABLE INITIALLY DEFERRED,
    CONSTRAINT fk_folders_uid_users FOREIGN KEY (uid)
        REFERENCES mail.users ON DELETE RESTRICT,
    CONSTRAINT check_position CHECK (
        position >= 0
    )
);

COMMENT ON COLUMN mail.folders.first_unseen
    IS 'IMAP unseen - sequence number of first unseen message in folder';
COMMENT ON COLUMN mail.folders.first_unseen_id
    IS 'IMAP unseen - imap_id of first unseen message in folder';


CREATE UNIQUE INDEX uk_folders_uid_type_is_unique_type
    ON mail.folders (uid, type)
 WHERE unique_type = true;


CREATE UNIQUE INDEX uk_folders_uid_name_parent_fid
    ON mail.folders (uid, name, COALESCE(parent_fid, 0));

-- Partial index for fast checking for folder is POP3 initialized
CREATE INDEX i_folders_uid_pop3state_is_enabled_or_initialized
    ON mail.folders (uid, pop3state, fid)
WHERE (pop3state).enabled = true OR (pop3state).initialized = true;

CREATE TABLE mail.imap_unsubscribed_folders (
    uid        bigint NOT NULL,
    full_name  text[] NOT NULL,
    revision   bigint NOT NULL,
    CONSTRAINT check_plain_not_empty_full_name CHECK (
            array_ndims(full_name) = 1
        AND cardinality(full_name) > 0
    ),
    CONSTRAINT check_no_nulls_and_empty_names_in_full_name CHECK (
            ('' != ALL(full_name)) IS NOT NULL
        AND
            '' != ALL(full_name)
    ),
    CONSTRAINT pk_imap_unsubscribed_folders PRIMARY KEY (uid, full_name),
    CONSTRAINT fk_imap_unsubscribed_folders FOREIGN KEY (uid)
        REFERENCES mail.users
);

CREATE TABLE mail.shared_folders (
    uid         bigint   NOT NULL,
    fid         integer  NOT NULL,
    revision    bigint   NOT NULL,
    created     timestamp with time zone NOT NULL DEFAULT current_timestamp,
    CONSTRAINT pk_shared_folders PRIMARY KEY (uid, fid),
    CONSTRAINT fk_shared_folders_folders
        FOREIGN KEY (uid, fid)
        REFERENCES mail.folders (uid, fid) ON DELETE NO ACTION
);

COMMENT ON TABLE mail.shared_folders
    IS 'make sense on shard where live shared folder owner';


CREATE TYPE mail.folder_archivation_type AS ENUM (
    'archive',
    'clean'
);

CREATE TABLE mail.folder_archivation_rules (
    uid             bigint   NOT NULL,
    fid             integer  NOT NULL,
    revision        bigint   NOT NULL,
    archive_type    mail.folder_archivation_type   NOT NULL,
    keep_days       integer  NOT NULL,
    created         timestamp with time zone NOT NULL DEFAULT current_timestamp,
    max_size        integer  NOT NULL DEFAULT constants.folder_max_size(),
    CONSTRAINT check_keep_days CHECK (
        keep_days > 0
    ),
    CONSTRAINT pk_folder_archivation_rules PRIMARY KEY (uid, fid),
    CONSTRAINT fk_folder_archivation_rules_folders
        FOREIGN KEY (uid, fid)
        REFERENCES mail.folders (uid, fid) ON DELETE NO ACTION,
    CONSTRAINT check_max_size CHECK (
        max_size > 0
    )
);

COMMENT ON COLUMN mail.folder_archivation_rules.max_size
     IS 'maximum amount of messages allowed for folder';

CREATE INDEX i_folder_archivation_rules_type
    ON mail.folder_archivation_rules (archive_type);


CREATE TABLE mail.doberman_jobs (
    worker_id      text NOT NULL,
    launch_id      text,
    heartbeated    timestamp with time zone,
    assigned       timestamp with time zone,
    hostname       text,
    worker_version text,
    timeout        interval,
    CONSTRAINT check_worker_id_not_empty CHECK (length(worker_id) > 0),
    CONSTRAINT pk_doberman_jobs PRIMARY KEY (worker_id)
);

CREATE SEQUENCE mail.doberman_jobs_change_log_log_id_seq START WITH 1 INCREMENT BY 1;

CREATE TABLE mail.doberman_jobs_change_log (
    log_id    bigint NOT NULL DEFAULT nextval('mail.doberman_jobs_change_log_log_id_seq'),
    log_date  timestamp with time zone NOT NULL DEFAULT current_timestamp,
    worker_id text  NOT NULL,
    info      jsonb NOT NULL
) PARTITION BY RANGE (log_date);

CREATE TABLE mail.doberman_jobs_change_log_templ (
    LIKE mail.doberman_jobs_change_log INCLUDING DEFAULTS
);

SELECT create_parent(
    p_parent_table := 'mail.doberman_jobs_change_log',
    p_template_table := 'mail.doberman_jobs_change_log_templ',
    p_control := 'log_date',
    p_type := 'native',
    p_interval := 'daily',
    p_premake := 9,
    p_jobmon := false
);
UPDATE part_config
   SET retention = '14 days',
       retention_keep_table = false,
       retention_keep_index = false
 WHERE parent_table = 'mail.doberman_jobs_change_log';


CREATE TYPE mail.subscription_state AS ENUM (
    'new',
    'init',
    'sync',
    'migrate',
    'discontinued',
    'terminated',
    'init-fail',
    'sync-fail',
    'migrate-fail',
    'migrate-finished',
    'clear',
    'clear-fail'
);

CREATE TABLE mail.shared_folder_subscriptions (
    uid             bigint NOT NULL,
    subscription_id bigint NOT NULL,
    fid             int    NOT NULL,
    subscriber_uid  bigint NOT NULL,
    created         timestamp with time zone NOT NULL DEFAULT current_timestamp,
    updated         timestamp with time zone NOT NULL DEFAULT current_timestamp,
    worker_id       text,
    state           mail.subscription_state NOT NULL DEFAULT 'new',
    fail_reason     text,
    CONSTRAINT pk_shared_folder_subscriptions
        PRIMARY KEY (uid, subscription_id),
    CONSTRAINT uk_shared_folder_subscriptions
        UNIQUE (uid, fid, subscriber_uid),
    CONSTRAINT fk_shared_folder_subscriptions_to_shared_folders
        FOREIGN KEY (uid, fid)
        REFERENCES mail.shared_folders (uid, fid) ON DELETE NO ACTION,
    CONSTRAINT fk_shared_folders_worker_id_to_doberman_jobs
        FOREIGN KEY (worker_id)
        REFERENCES mail.doberman_jobs (worker_id) ON DELETE NO ACTION
);

CREATE INDEX i_shared_folder_subscriptions_worker_id
    ON mail.shared_folder_subscriptions (worker_id);


CREATE TABLE mail.subscribed_folders (
    uid             bigint  NOT NULL,
    fid             integer NOT NULL,
    revision        bigint  NOT NULL,
    owner_uid       bigint  NOT NULL,
    owner_fid       integer NOT NULL,
    subscription_id bigint  NOT NULL,
    synced_revision bigint  NOT NULL,
    created         timestamp with time zone NOT NULL DEFAULT current_timestamp,
    synced_imap_id  bigint,
    CONSTRAINT pk_subscribed_folders
        PRIMARY KEY (uid, subscription_id),
    CONSTRAINT uk_subscribed_folders_uid_fid
        UNIQUE (uid, fid),
    CONSTRAINT uk_subscribed_folders_uid_owner_uid_fid
        UNIQUE (uid, owner_uid, owner_fid),
    CONSTRAINT fk_subscribed_folders_folders
        FOREIGN KEY (uid, fid)
        REFERENCES mail.folders (uid, fid) ON DELETE NO ACTION
);

COMMENT ON TABLE mail.subscribed_folders
     IS 'make sense on shard where live subscriber';

COMMENT ON COLUMN mail.subscribed_folders.revision
     IS 'latest synced revision in subscriber terms';

COMMENT ON COLUMN mail.subscribed_folders.synced_revision
     IS 'latest synced revision in owner terms';

COMMENT ON COLUMN mail.subscribed_folders.synced_imap_id
     IS 'latest synced message imap_id in owner terms';

CREATE TABLE mail.unsubscribe_tasks (
    task_id                 bigserial,
    task_request_id         text NOT NULL,
    owner_uid               bigint NOT NULL,
    owner_fids              integer[] NOT NULL,
    subscriber_uid          bigint NOT NULL,
    root_subscriber_fid     integer NOT NULL,
    assigned                timestamp with time zone DEFAULT NULL,
    CONSTRAINT check_owner_fids_sorted CHECK (
        owner_fids = uniq(sort(owner_fids))
    ),
    CONSTRAINT pk_unsubscribe_tasks
        PRIMARY KEY (task_id),
    CONSTRAINT uk_unsubscribe_tasks_owner_subscriber_fids
        UNIQUE (owner_uid, owner_fids, subscriber_uid, root_subscriber_fid),
    CONSTRAINT fk_unsubscribe_tasks_users
        FOREIGN KEY (owner_uid)
        REFERENCES mail.users (uid) ON DELETE CASCADE
);

CREATE INDEX i_unsubscribe_tasks_assigned
    ON mail.unsubscribe_tasks (assigned);

CREATE TYPE mail.labels_types AS ENUM (
    'user',
    'system',
    'domain',
    'type',
    'imap',
    'rpop'
);

CREATE TABLE mail.labels (
    uid      bigint  NOT NULL,
    lid      integer NOT NULL,
    revision bigint  NOT NULL,
    name     text    NOT NULL,
    type     mail.labels_types NOT NULL,
    color    text,
    created  timestamp with time zone NOT NULL DEFAULT current_timestamp,

    message_count integer DEFAULT 0 NOT NULL,
    message_seen integer DEFAULT 0,
    CONSTRAINT check_message_seen CHECK (
        message_seen is NULL OR (message_seen >= 0 AND message_count >= message_seen)
    ),
    CONSTRAINT check_message_count CHECK (
        message_count >= 0
    ),
    CONSTRAINT check_non_empty_name CHECK (
        name != ''
    ),
    CONSTRAINT check_so_name CHECK (
        type <> 'type'::mail.labels_types
     OR name ~ '^[\d]+$'
    ),
    CONSTRAINT pk_labels PRIMARY KEY (uid, lid),
    CONSTRAINT uk_labels_uid_type_name UNIQUE (uid, type, name),
    CONSTRAINT fk_labels_uid_users FOREIGN KEY (uid)
        REFERENCES mail.users ON DELETE RESTRICT
);


CREATE TYPE mail.tab_types AS ENUM (
    'relevant',
    'news',
    'social'
);

CREATE TABLE mail.tabs (
    uid            bigint  NOT NULL,
    tab            mail.tab_types  NOT NULL,
    revision       bigint  NOT NULL,
    unvisited      boolean  NOT NULL DEFAULT 'no',
    created        timestamp with time zone NOT NULL DEFAULT current_timestamp,

    message_count  integer NOT NULL DEFAULT 0,
    message_seen   integer NOT NULL DEFAULT 0,
    message_size   bigint  NOT NULL DEFAULT 0,
    attach_count   integer NOT NULL DEFAULT 0,
    attach_size    bigint  NOT NULL DEFAULT 0,
    fresh_count    integer NOT NULL DEFAULT 0,

    CONSTRAINT check_message_count CHECK (
        message_count >= 0 AND
        message_seen  >= 0 AND
        message_count >= message_seen
    ),
    CONSTRAINT check_message_size CHECK (
        (message_size > 0 AND message_count > 0)
        OR
        (message_size = 0 AND message_count = 0)
    ),
    CONSTRAINT check_attaches CHECK (
        attach_count >= 0 AND
        attach_size >= 0 AND
        attach_size <= message_size
    ),
    CONSTRAINT check_revision CHECK (
        revision > 0
    ),
    CONSTRAINT pk_tabs PRIMARY KEY (uid, tab),
    CONSTRAINT fk_tabs_uid_users FOREIGN KEY (uid)
        REFERENCES mail.users ON DELETE RESTRICT
);


CREATE TYPE mail.recipient_type AS ENUM (
    'from',
    'to',
    'reply-to',
    'cc',
    'bcc',
    'reply-to-all',
    'sender'
);

CREATE TYPE mail.recipient AS (
    type  mail.recipient_type,
    name  text,
    email text
);

CREATE TYPE mail.attach AS (
    hid         text,
    type        text,
    filename    text,
    size        integer
);

CREATE TYPE mail.mime_part AS (
    hid                 text,
    content_type        text,
    content_subtype     text,
    boundary            text,
    name                text,
    charset             text,
    encoding            text,
    content_disposition text,
    filename            text,
    cid                 text,

    offset_begin        integer,
    offset_end          integer
);

CREATE TYPE mail.message_attributes AS ENUM (
     'spam',
     'postmaster',
     'mulca-shared',
     'sms',
     'append',
     'copy',
     'synced'
);

CREATE TYPE mail.threads_merge_rules AS ENUM (
    'hash',
    'references',
    'force-new-thread',
    'undefined',
    'synced'
);

CREATE TABLE mail.messages (
    uid   bigint  NOT NULL,
    mid   bigint  NOT NULL,
    st_id text    NOT NULL,
    size  integer NOT NULL,

    attributes      mail.message_attributes[]
        CHECK (array_ndims(attributes) = 1),
    attaches        mail.attach[]
        CHECK (array_ndims(attaches) = 1),
    subject         text,
    firstline       text,
    hdr_date        timestamp with time zone,
    hdr_message_id  text,
    recipients      mail.recipient[]
        CHECK (array_ndims(recipients) = 1),
    extra_data      text, -- reply_to_all

    pop_uidl        text,
    found_tid       bigint,
    thread_rule     mail.threads_merge_rules,
    mime            mail.mime_part[],

    CONSTRAINT check_plain_not_empty_mime CHECK (
            array_ndims(mime) = 1
        AND cardinality(mime) > 0
	),
    CONSTRAINT pk_messages PRIMARY KEY (uid, mid),
    CONSTRAINT check_size CHECK (
        size >= 0
    ),
    CONSTRAINT fk_messages_users_uid FOREIGN KEY (uid)
        REFERENCES mail.users ON DELETE RESTRICT,
    CONSTRAINT messages_st_id_without_mulca_2 CHECK (st_id NOT LIKE 'mulca:2:%')
);

-- maybe create index on hash(hdr_message_id)
CREATE INDEX i_messages_uid_message_id
    ON mail.messages (uid, hdr_message_id);

CREATE INDEX if_messages_hashtext_st_id
    ON mail.messages (hashtext(st_id));

CREATE INDEX i_messages_has_attaches
    ON mail.messages (uid, mid)
 WHERE cardinality(attaches) > 0;

CREATE TABLE mail.windat_messages (
    uid     bigint         NOT NULL,
    mid     bigint         NOT NULL,
    st_id   text           NOT NULL,
    hid     text           NOT NULL,
    windat  mail.mime_part,

    CONSTRAINT pk_windat_messages PRIMARY KEY (uid, mid, hid),
    CONSTRAINT check_windat_messages_hid CHECK (hid IS NOT DISTINCT FROM (windat).hid),
    CONSTRAINT fk_windat_messages_uid_mid FOREIGN KEY (uid, mid)
        REFERENCES mail.messages (uid, mid),
    CONSTRAINT check_windat_messages_stid CHECK (st_id ~ '^\d+.windat.E\d+:\d+$')
);

COMMENT ON COLUMN mail.windat_messages.st_id
    IS 'st_id of windat attach != st_id of message';

COMMENT ON COLUMN mail.windat_messages.hid
    IS 'hid of windat attach = hid of windat part + .1, .2, .3, ...';

CREATE INDEX if_windat_messages_hashtext_st_id
    ON mail.windat_messages (hashtext(st_id));

CREATE TABLE mail.box (
    uid           bigint  NOT NULL,
    mid           bigint  NOT NULL,
    fid           integer NOT NULL,
    tid           bigint,
    newest_tif    boolean NOT NULL DEFAULT 'no',
    imap_id       bigint NOT NULL,
    revision      bigint NOT NULL,
    chain         smallint,
    seen          boolean NOT NULL,
    recent        boolean NOT NULL,
    deleted       boolean NOT NULL,
    received_date timestamp with time zone NOT NULL,
    lids          integer[] NOT NULL
        CHECK (array_ndims(lids) = 1),
    doom_date     timestamp with time zone,

    tab           mail.tab_types,
    newest_tit    boolean,

    CONSTRAINT check_duplicate_lids CHECK (
        lids IS NULL OR (#lids = #uniq(sort(lids)))
    ),
    CONSTRAINT check_mid CHECK (
        mid > 0
    ),
    CONSTRAINT check_chain CHECK (
        chain IS NULL OR chain > 0
    ),
    CONSTRAINT check_newest_on_not_threaded CHECK (
        (tid IS NULL AND NOT newest_tif)
        OR
        tid IS NOT NULL
    ),
    CONSTRAINT check_newest_tit_on_not_tabbed CHECK (
        (tab IS NULL AND newest_tit IS NULL)
        OR
        (tab IS NOT NULL AND newest_tit IS NOT NULL)
    ),
    CONSTRAINT check_newest_tit_on_not_threaded CHECK (
        (tid IS NULL AND (newest_tit IS NULL OR NOT newest_tit))
        OR
        tid IS NOT NULL
    ),
    CONSTRAINT pk_box PRIMARY KEY (uid, mid),
    CONSTRAINT uk_box_imap_id UNIQUE (uid, fid, imap_id),
    CONSTRAINT fk_box_messages_uid_mid FOREIGN KEY (uid, mid)
        REFERENCES mail.messages ON DELETE RESTRICT INITIALLY DEFERRED
);

CREATE INDEX i_box_uid_fid_received_date_desc_flags
    ON mail.box (uid, fid, received_date DESC, seen);
CREATE INDEX i_box_uid_tid
    ON mail.box (uid, tid);
CREATE INDEX ip_box_is_newest_uid_fid_received_date
    ON mail.box (uid, fid, received_date DESC)
 WHERE newest_tif = true;
CREATE INDEX i_box_uid_fid_revision
    ON mail.box (uid, fid, revision);
CREATE INDEX i_box_uid_fid_imap_id_chain
    ON mail.box (uid, fid, imap_id, chain)
 WHERE chain IS NOT NULL;
CREATE INDEX i_box_uid_fid_deleted
    ON mail.box (uid, fid)
 WHERE deleted = true;
CREATE INDEX i_box_doom_date_uid_fid
    ON mail.box (doom_date, uid, fid)
 WHERE doom_date IS NOT NULL;
CREATE INDEX i_box_uid_tab_received_date_desc_flags
    ON mail.box (uid, tab, received_date DESC, seen)
 WHERE tab IS NOT NULL;
CREATE INDEX ip_box_is_newest_uid_tab_received_date
    ON mail.box (uid, tab, received_date DESC)
 WHERE tab IS NOT NULL AND newest_tit = true;

CREATE OR REPLACE FUNCTION mail.ulids(
    i_uid  bigint,
    i_lids integer[]
) RETURNS bigint[] AS $$
    SELECT array_agg((i_uid << 32) | lid)
      FROM unnest(i_lids) lid;
$$ LANGUAGE SQL IMMUTABLE STRICT;

CREATE INDEX ip_box_uid_lids
    ON mail.box USING gin (mail.ulids(uid, lids)) WITH (FASTUPDATE=OFF)  WHERE cardinality(lids) > 0;

ALTER TABLE mail.box ADD CONSTRAINT fk_box_folders_fid
    FOREIGN KEY (uid, fid) REFERENCES mail.folders ON DELETE RESTRICT;

CREATE TABLE mail.deleted_box (
    deleted_date  timestamp with time zone DEFAULT CURRENT_TIMESTAMP,
    uid           bigint NOT NULL,
    mid           bigint NOT NULL,
    revision      bigint NOT NULL,
    info          jsonb NOT NULL,
    received_date timestamp with time zone DEFAULT NULL,
    CONSTRAINT pk_deleted_box PRIMARY KEY (uid, mid),
    CONSTRAINT fk_deleted_box_messages_uid_mid
        FOREIGN KEY (uid, mid)
        REFERENCES mail.messages (uid, mid)
        ON DELETE RESTRICT
);
CREATE INDEX i_deleted_box_uid_deleted_date_desc
    ON mail.deleted_box (uid, deleted_date DESC);

CREATE TABLE mail.pop3_box (
    uid      bigint  NOT NULL,
    mid      bigint  NOT NULL,
    fid      integer NOT NULL,
    size     integer NOT NULL,
    old_uidl text,
    CONSTRAINT pk_pop3_box PRIMARY KEY (uid, mid),
    CONSTRAINT fk_pop3_box_box FOREIGN KEY (uid, mid)
        REFERENCES mail.box ON DELETE NO ACTION DEFERRABLE INITIALLY DEFERRED,
    CONSTRAINT fk_pop3_box_folders FOREIGN KEY (uid, fid)
        REFERENCES mail.folders ON DELETE CASCADE
);

CREATE INDEX i_pop3_box_uid_fid
    ON mail.pop3_box (uid, fid);

COMMENT ON COLUMN mail.pop3_box.fid
    IS 'mail.box.fid replica';
COMMENT ON COLUMN mail.pop3_box.size
    IS 'mail.messages.size replica';
COMMENT ON COLUMN mail.pop3_box.old_uidl
    IS 'used only for messages transfered from oracle, for new messages we calculate it from uid and mid';


CREATE TABLE mail.synced_messages (
    uid            bigint  NOT NULL,
    mid            bigint  NOT NULL,
    tid            bigint  NOT NULL,
    revision       bigint  NOT NULL,
    subscription_id bigint NOT NULL,
    owner_mid      bigint  NOT NULL,
    owner_tid      bigint  NOT NULL,
    owner_revision bigint  NOT NULL,
    CONSTRAINT pk_synced_messages PRIMARY KEY (uid, mid),
    CONSTRAINT uk_synced_messages_uid_subscription_id_owner_mid
        UNIQUE (uid, subscription_id, owner_mid),
    CONSTRAINT fk_synced_messages_box
        FOREIGN KEY (uid, mid)
        REFERENCES mail.box (uid, mid)
        ON DELETE CASCADE,
    CONSTRAINT fk_synced_messages_subscribed_folders
        FOREIGN KEY (uid, subscription_id)
        REFERENCES mail.subscribed_folders (uid, subscription_id)
        ON DELETE CASCADE
);

CREATE INDEX i_synced_messages_uid_subscription_id_owner_tid
    ON mail.synced_messages (uid, subscription_id, owner_tid);


CREATE TYPE mail.change_type AS ENUM (
    'store',
    'update',
    'move',
    'delete',
    'copy',
    'threads-join',
    'label-create',
    'label-delete',
    'label-modify',
    'folder-create',
    'folder-delete',
    'folder-modify',
    'fresh-reset',
    'transfer',
    'update-attach',
    'folder-modify-type',
    'quick-save',
    'reindex',
    'imap-add-unsubscribed',
    'imap-delete-unsubscribed',
    'pop3-folders-enable',
    'pop3-folders-disable',
    'pop3-folder-initialization',
    'pop3-delete',
    'register',
    'user-delete',
    'shared-folder-create',
    'shared-folder-delete', -- currently not used
    'shared-folder-subscribe',
    'shared-folder-unsubscribe',
    'subscribed-folder-create',
    'subscribed-folder-delete',
    'sync-store',
    'sync-delete',
    'sync-update',
    'folder-reset-unvisited',
    'sync-threads-join',
    'message-storage-change',
    'set-archivation-rule',
    'remove-archivation-rule',
    'create-contacts-user',
    'tab-create',
    'tab-reset-unvisited',
    'move-to-tab',
    'collector-create',
    'collector-delete',
    'collector-update',
    'user-state-update',
    'store-deleted',
    'sticker-create',
    'sticker-remove'
);

CREATE SEQUENCE mail.cid_seq START WITH 1 INCREMENT BY 1 CYCLE;

CREATE TABLE mail.change_log (
    cid              bigint           NOT NULL DEFAULT nextval('mail.cid_seq'),
    uid              bigint           NOT NULL,
    revision         bigint           NOT NULL,
    type             mail.change_type NOT NULL,
    change_date      timestamp with time zone NOT NULL DEFAULT current_timestamp,
    changed          jsonb,
    arguments        jsonb,
    fresh_count      bigint,
    useful_new_count bigint,
    db_user          text             DEFAULT CURRENT_USER,
    x_request_id    text,
    session_key     text,
    quiet           boolean
) PARTITION BY RANGE (change_date);

CREATE TABLE mail.change_log_templ (
    LIKE mail.change_log INCLUDING DEFAULTS
);

ALTER TABLE mail.change_log_templ
    ADD CONSTRAINT pk_change_log_templ PRIMARY KEY (cid);

CREATE INDEX i_change_log_uid_revision_templ
    ON mail.change_log_templ (uid, revision);

SELECT create_parent(
    p_parent_table := 'mail.change_log',
    p_template_table := 'mail.change_log_templ',
    p_control := 'change_date',
    p_type := 'native',
    p_interval := 'daily',
    p_premake := 9,
    p_jobmon := false
);
UPDATE part_config
   SET retention = '30 days',
       retention_keep_table = false,
       retention_keep_index = false
 WHERE parent_table = 'mail.change_log';


CREATE TABLE mail.shared_folder_change_queue (
    uid             bigint NOT NULL,
    subscription_id bigint NOT NULL,
    cid             bigint NOT NULL,
    CONSTRAINT pk_shared_folder_change_queue
        PRIMARY KEY (uid, subscription_id, cid),
    CONSTRAINT fk_shared_folders_change_queue_to_subscriptions
        FOREIGN KEY (uid, subscription_id)
        REFERENCES mail.shared_folder_subscriptions (uid, subscription_id)
        ON DELETE CASCADE
);


CREATE TABLE mail.chained_log (
    uid        bigint   NOT NULL,
    fid        integer  NOT NULL,
    revision   bigint   NOT NULL,
    mid        bigint   NOT NULL,
    chained_id bigint   NOT NULL,
    imap_id    bigint   NOT NULL,
    distance   smallint NOT NULL,
    log_date   timestamp with time zone NOT NULL DEFAULT current_timestamp,
    CONSTRAINT pk_chained_log PRIMARY KEY (uid, fid, revision, mid),
    CONSTRAINT fk_chained_log_uid_fid_folders FOREIGN KEY (uid, fid)
        REFERENCES mail.folders ON DELETE CASCADE
);

COMMENT ON COLUMN mail.chained_log.chained_id IS
    'chained imap_id for this mid';
COMMENT ON COLUMN mail.chained_log.imap_id IS
    'imap_id, but 0 in case of chained change';
COMMENT ON COLUMN mail.chained_log.distance IS
    'message count between chained and this imap_id, but chain size if case of new chained';

CREATE SEQUENCE mail.fix_id_seq START WITH 1 INCREMENT BY 1 CYCLE;

CREATE TABLE mail.fix_log (
    uid      bigint NOT NULL,
    revision bigint NOT NULL,
    fix_key  text   NOT NULL,
    fixed    jsonb  NOT NULL,
    fix_date timestamp with time zone NOT NULL DEFAULT current_timestamp,
    db_user  text   NOT NULL DEFAULT current_user,
    fix_id   bigint NOT NULL DEFAULT nextval('mail.fix_id_seq'),
    CONSTRAINT pk_fix_log PRIMARY KEY (uid, fix_id, fix_date),
    CONSTRAINT fk_fix_log_uid_users FOREIGN KEY (uid)
        REFERENCES mail.users ON DELETE CASCADE
);


CREATE TYPE mail.thread_label AS (
    lid            integer,
    message_count  integer
);

CREATE TABLE mail.threads (
    uid         bigint NOT NULL,
    tid         bigint NOT NULL,
    revision    bigint NOT NULL,
    newest_mid  bigint NOT NULL,
    newest_date timestamp with time zone NOT NULL,

    message_count integer NOT NULL,
    message_seen  integer NOT NULL,

    labels mail.thread_label[]
        CHECK (array_ndims(labels) = 1),

    CONSTRAINT check_message_count CHECK (
        message_count > 0 and
        message_seen  >= 0 and
        message_count >= message_seen
    ),
    attach_count integer NOT NULL DEFAULT 0,
    attach_size  bigint NOT NULL DEFAULT 0,
    CONSTRAINT check_attaches CHECK (
        attach_count >= 0 and
        attach_size  >= 0
    ),
    CONSTRAINT pk_threads PRIMARY KEY (uid, tid),
    CONSTRAINT fk_threads_newest_mid_box FOREIGN KEY (uid, newest_mid)
        REFERENCES mail.box ON DELETE NO ACTION DEFERRABLE INITIALLY DEFERRED
);

CREATE UNIQUE INDEX uk_threads_uid_newest_mid
    ON mail.threads (uid, newest_mid);

CREATE OR REPLACE FUNCTION mail.uid_thread_labels(
    i_uid  bigint,
    i_thread_labels mail.thread_label[]
) RETURNS bigint[] AS $$
    SELECT array_agg((i_uid << 32) | thr_label.lid)
      FROM unnest(i_thread_labels) thr_label;
$$ LANGUAGE SQL IMMUTABLE STRICT;

CREATE INDEX ip_uid_thread_labels
    ON mail.threads USING gin (mail.uid_thread_labels(uid, labels)) WITH (FASTUPDATE=OFF)  WHERE cardinality(labels) > 0;

CREATE TYPE mail.thread_hash_namespaces AS ENUM (
    'subject',
    'from'
);

CREATE TABLE mail.threads_hashes (
    uid        bigint NOT NULL,
    namespace  mail.thread_hash_namespaces NOT NULL,
    value      numeric(20, 0) NOT NULL,
    uniq_key   bigint NOT NULL,
    tid        bigint NOT NULL,
    CONSTRAINT pk_threads_hashes PRIMARY KEY (uid, namespace, value, uniq_key),
    CONSTRAINT fk_threads_hashes_threads FOREIGN KEY (uid, tid)
        REFERENCES mail.threads ON DELETE CASCADE
);

CREATE INDEX i_threads_hashes_uid_tid
    ON mail.threads_hashes (uid, tid);


CREATE TYPE mail.message_reference_type AS ENUM (
    'reference',
    'in-reply-to'
);

CREATE TABLE mail.message_references (
    uid   bigint NOT NULL,
    mid   bigint NOT NULL,
    value numeric(20, 0) NOT NULL,
    type  mail.message_reference_type NOT NULL,
    CONSTRAINT pk_message_references PRIMARY KEY (uid, value, mid),
    CONSTRAINT fk_message_references_messages FOREIGN KEY (uid, mid)
        REFERENCES mail.messages ON DELETE CASCADE INITIALLY DEFERRED
);

CREATE INDEX i_message_references_uid_mid
    ON mail.message_references (uid, mid);

CREATE TABLE mail.threads_messages (
    uid          bigint NOT NULL,
    mid          bigint NOT NULL,
    tid          bigint NOT NULL,
    rule         mail.threads_merge_rules NOT NULL,
    sort_options text,
    CONSTRAINT pk_threads_messages PRIMARY KEY (uid, mid),
    CONSTRAINT fk_threads_messages_messages FOREIGN KEY (uid, mid)
        REFERENCES mail.messages ON DELETE CASCADE
);

CREATE SEQUENCE mail.local_transfer_id_seq START WITH 1 INCREMENT BY 1 CYCLE;

CREATE TABLE mail.transfer_info (
    from_db           text,
    from_uid          bigint,
    from_suid         bigint,
    to_db             text,
    to_uid            bigint,
    to_suid           bigint,
    transfer_start    timestamp with time zone NOT NULL,
    transfer_end      timestamp with time zone NOT NULL,
    script            text NOT NULL,
    script_revision   text NOT NULL,
    transfered_at     text NOT NULL,
    local_transfer_id bigint NOT NULL DEFAULT nextval('mail.local_transfer_id_seq'),
    CONSTRAINT pk_transfer_info PRIMARY KEY (local_transfer_id)
);
CREATE INDEX i_transfer_info_from_uid_to_uid
    ON mail.transfer_info (from_uid, to_uid);

CREATE TABLE mail.counters (
    uid         bigint  NOT NULL,
    fresh_count integer NOT NULL    DEFAULT 0,
    revision    bigint  NOT NULL,

    has_attaches_count  integer  DEFAULT 0,
    has_attaches_seen   integer  DEFAULT 0,

    CONSTRAINT pk_counters PRIMARY KEY (uid),
    CONSTRAINT check_fresh_count CHECK (
        fresh_count >= 0
    ),
    CONSTRAINT fk_counters_uid_users FOREIGN KEY (uid)
        REFERENCES mail.users ON DELETE CASCADE
);

ALTER TABLE mail.users
    ADD CONSTRAINT fk_users_uid_counters
    FOREIGN KEY (uid) REFERENCES mail.counters
    DEFERRABLE INITIALLY DEFERRED;

CREATE SEQUENCE mail.sdq_seq START WITH 1 INCREMENT BY 1 CYCLE;

CREATE TABLE mail.storage_delete_queue (
    deleted_date  timestamp with time zone DEFAULT CURRENT_TIMESTAMP,
    uid           bigint NOT NULL,
    st_id         text NOT NULL,
    qid           bigint NOT NULL DEFAULT nextval('mail.sdq_seq'),
    fails_count   integer NOT NULL DEFAULT 0,
    CONSTRAINT storage_delete_queue_st_id_without_mulca_2
        CHECK (st_id NOT LIKE 'mulca:2:%'),
    CONSTRAINT pk_storage_delete_queue PRIMARY KEY (qid)
);
CREATE INDEX i_storage_delete_queue_deleted_date
    ON mail.storage_delete_queue (deleted_date);
CREATE INDEX i_storage_delete_queue_uid_st_id
    ON mail.storage_delete_queue (uid, st_id);

CREATE INDEX brin_chained_log_log_date
    ON mail.chained_log USING BRIN (log_date);

CREATE TABLE mail.collectors (
    uid          bigint  NOT NULL,
    collector_id integer NOT NULL,
    metadata     jsonb   NOT NULL,

    CONSTRAINT pk_collectors PRIMARY KEY (uid, collector_id),
    CONSTRAINT fk_collectors_uid FOREIGN KEY (uid)
        REFERENCES mail.users ON DELETE RESTRICT
);

CREATE SCHEMA mailish;

CREATE TABLE mailish.accounts (
    uid         bigint  NOT NULL,
    email       text    NOT NULL,
    imap_server text    NOT NULL,
    imap_port   integer NOT NULL,
    smtp_server text    NOT NULL,
    smtp_port   integer NOT NULL,
    last_sync   timestamp with time zone,
    imap_login  text    DEFAULT NULL,
    imap_ssl    boolean DEFAULT NULL,
    smtp_login  text    DEFAULT NULL,
    smtp_ssl    boolean DEFAULT NULL,

    CONSTRAINT pk_mailish_accounts PRIMARY KEY(uid),
    CONSTRAINT fk_mailish_accounts_uid FOREIGN KEY(uid)
        REFERENCES mail.users ON DELETE CASCADE
);

CREATE TYPE mailish.auth_type AS ENUM (
    'oauth2',
    'password'
);

CREATE TABLE mailish.auth_data (
    uid         bigint  NOT NULL,
    token_id    text    NOT NULL,
    auth_type   mailish.auth_type NOT NULL,
    uuid        text    NOT NULL,
    lock_flag   boolean NOT NULL DEFAULT TRUE,
    last_valid  timestamp with time zone DEFAULT now(),
    oauth_app   text    DEFAULT NULL,
    imap_credentials text DEFAULT NULL,
    smtp_credentials text DEFAULT NULL,

    CONSTRAINT pk_mailish_auth_data PRIMARY KEY(uid, token_id),
    CONSTRAINT fk_mailish_auth_data_uid FOREIGN KEY (uid)
        REFERENCES mailish.accounts ON DELETE CASCADE
);

CREATE TABLE mailish.folders (
    uid         bigint  NOT NULL,
    fid         integer NOT NULL,
    imap_path   text    NOT NULL,
    uidvalidity bigint  NOT NULL,
    range_start bigint  DEFAULT 0,
    range_end   bigint  DEFAULT 0,

    CONSTRAINT pk_mailish_folders PRIMARY KEY (uid, imap_path),
    CONSTRAINT fk_mailish_folders_uid_fid FOREIGN KEY (uid, fid)
        REFERENCES mail.folders ON DELETE CASCADE,
    CONSTRAINT fk_mailish_folders_uid FOREIGN KEY(uid)
        REFERENCES mailish.accounts ON DELETE CASCADE,

    CONSTRAINT not_empty_imap_path CHECK (length(imap_path) > 0),
    CONSTRAINT valid_downloaded_range CHECK (range_start <= range_end),

    CONSTRAINT uk_mailish_folders UNIQUE (uid, fid)
);

CREATE TABLE mailish.messages (
    uid         bigint  NOT NULL,
    fid         integer NOT NULL,
    imap_id     bigint  NOT NULL,
    imap_time   timestamp with time zone,
    mid         bigint,
    errors      integer DEFAULT 0,

    CONSTRAINT pk_mailish_messages PRIMARY KEY (uid, fid, imap_id),
    CONSTRAINT fk_mailish_messages_uid_mid FOREIGN KEY (uid, mid)
        REFERENCES mail.box ON DELETE CASCADE,
    CONSTRAINT fk_mailish_messages_uid_fid FOREIGN KEY (uid, fid)
        REFERENCES mailish.folders (uid, fid) ON DELETE CASCADE
);

CREATE INDEX ix_mailish_messages_uid_mid ON mailish.messages (uid, mid);


CREATE TYPE mail.archive_state AS ENUM (
    'archivation_in_progress',
    'archivation_complete',
    'archivation_error',
    'restoration_in_progress',
    'restoration_complete',
    'restoration_error',
    'restoration_requested',
    'cleaning_requested',
    'cleaning_in_progress'
);

CREATE TABLE mail.archives (
    uid                         bigint NOT NULL,

    version                     integer NOT NULL DEFAULT 1,
    state                       mail.archive_state NOT NULL DEFAULT 'archivation_in_progress',
    revision                    bigint,

    message_count               integer NOT NULL DEFAULT 0,
    restored_message_count      integer NOT NULL DEFAULT 0,

    updated                     timestamptz NOT NULL DEFAULT current_timestamp,
    notice                      text,
    task_id                     bigint,

    CONSTRAINT check_task_id CHECK (
        task_id IS NULL OR task_id >= 0
    ),

    CONSTRAINT pk_archives PRIMARY KEY (uid),
    CONSTRAINT fk_archives_users FOREIGN KEY (uid) REFERENCES mail.users ON DELETE CASCADE,

    CONSTRAINT check_counters CHECK (
        message_count >= 0 AND
        restored_message_count >= 0 
    )
);

CREATE TABLE mail.stickers_reply_later (
    uid                         bigint NOT NULL,
    mid                         bigint NOT NULL,
    fid                         integer NOT NULL,
    date                        timestamptz NOT NULL,
    created                     timestamptz NOT NULL DEFAULT current_timestamp,
    tab                         mail.tab_types,

    CONSTRAINT pk_stickers_reply_later PRIMARY KEY (uid, mid),
    CONSTRAINT fk_stickers_reply_later_box FOREIGN KEY (uid, mid) REFERENCES mail.box ON DELETE CASCADE
);

