CREATE SCHEMA IF NOT EXISTS contacts;

CREATE TYPE contacts.user_type AS ENUM (
    'passport_user',
    'passport_domain',
    'connect_organization'
);

CREATE TABLE contacts.users (
    user_id bigint NOT NULL,
    user_type contacts.user_type NOT NULL,
    here_since timestamp with time zone NOT NULL DEFAULT current_timestamp,
    is_here boolean NOT NULL DEFAULT true,
    is_deleted boolean NOT NULL DEFAULT false,

    is_directory_sync_enabled boolean NOT NULL DEFAULT false,
    directory_synced_revision bigint NOT NULL DEFAULT 0,
    directory_last_event_id bigint NOT NULL DEFAULT 0,
    directory_last_synced_event_id bigint NOT NULL DEFAULT 0,
    directory_last_sync_date timestamp with time zone NOT NULL DEFAULT current_timestamp,
    directory_pending_events_count bigint NOT NULL DEFAULT 0,

    CONSTRAINT pk_users PRIMARY KEY (user_id, user_type),
    CONSTRAINT check_directory_synced_revision CHECK (directory_synced_revision >= 0),
    CONSTRAINT check_directory_last_event_id CHECK (directory_last_event_id >= directory_last_synced_event_id),
    CONSTRAINT check_directory_last_synced_event_id CHECK (directory_last_synced_event_id >= 0),
    CONSTRAINT check_directory_pending_events_count CHECK (directory_pending_events_count >= 0)
);

CREATE INDEX i_users_by_directory_pending_events_count
    ON contacts.users (user_id, user_type, directory_pending_events_count)
 WHERE is_directory_sync_enabled IS true;

CREATE INDEX i_users_by_directory_last_sync_date
    ON contacts.users (user_id, user_type, directory_last_sync_date)
 WHERE is_directory_sync_enabled IS true;

CREATE TABLE contacts.serials (
    user_id bigint NOT NULL,
    user_type contacts.user_type NOT NULL,
    next_list_id bigint NOT NULL DEFAULT 1,
    next_contact_id bigint NOT NULL DEFAULT 1,
    next_tag_id bigint NOT NULL DEFAULT 1,
    next_revision bigint NOT NULL DEFAULT 1,
    next_email_id bigint NOT NULL DEFAULT 1,

    CONSTRAINT pk_serials PRIMARY KEY (user_id, user_type),
    CONSTRAINT fk_serials_user_id_user_type_users FOREIGN KEY (user_id, user_type)
        REFERENCES contacts.users ON DELETE CASCADE,
    CONSTRAINT check_next_list_id CHECK (
        next_list_id > 0
    ),
    CONSTRAINT check_next_contact_id CHECK (
        next_contact_id > 0
    ),
    CONSTRAINT check_next_tag_id CHECK (
        next_tag_id > 0
    ),
    CONSTRAINT check_next_revision CHECK (
        next_revision > 0
    ),
    CONSTRAINT check_next_email_id CHECK (
        next_email_id > 0
    )
);

CREATE TYPE contacts.list_type AS ENUM (
    'personal',
    'user'
);

CREATE TABLE contacts.lists (
    user_id bigint NOT NULL,
    user_type contacts.user_type NOT NULL,
    list_id bigint NOT NULL,
    revision bigint NOT NULL,
    name text NOT NULL,
    type contacts.list_type NOT NULL,

    CONSTRAINT pk_lists PRIMARY KEY (user_id, user_type, list_id),
    CONSTRAINT fk_lists_user_id_user_type_users FOREIGN KEY (user_id, user_type)
        REFERENCES contacts.users ON DELETE CASCADE,
    CONSTRAINT check_list_id CHECK (
        list_id > 0
    ),
    CONSTRAINT check_revision CHECK (
        revision > 0
    ),
    CONSTRAINT check_non_empty_name CHECK (
        name <> ''
    )
);

CREATE UNIQUE INDEX uk_lists_name_and_type_for_user
    ON contacts.lists (user_id, user_type, name, type);

CREATE INDEX i_lists_user_id_user_type
    ON contacts.lists (user_id, user_type);

CREATE TYPE contacts.tag_type AS ENUM (
    'system',
    'user'
);

CREATE TABLE contacts.tags (
    user_id bigint NOT NULL,
    user_type contacts.user_type NOT NULL,
    tag_id bigint NOT NULL,
    revision bigint NOT NULL,
    name text NOT NULL,
    type contacts.tag_type NOT NULL,

    CONSTRAINT pk_tags PRIMARY KEY (user_id, user_type, tag_id),
    CONSTRAINT fk_tags_user_id_user_type_users FOREIGN KEY (user_id, user_type)
        REFERENCES contacts.users ON DELETE CASCADE,
    CONSTRAINT check_tag_id CHECK (
        tag_id > 0
    ),
    CONSTRAINT check_revision CHECK (
        revision > 0
    ),
    CONSTRAINT check_non_empty_name CHECK (
        name <> ''
    )
);

CREATE UNIQUE INDEX uk_tags_name_and_type_for_user
    ON contacts.tags (user_id, user_type, name, type);

CREATE INDEX i_tags_user_id_user_type
    ON contacts.tags (user_id, user_type);

CREATE TYPE contacts.vcard_format AS ENUM (
    'vcard_v1'
);

CREATE TABLE contacts.contacts (
    user_id bigint NOT NULL,
    user_type contacts.user_type NOT NULL,
    contact_id bigint NOT NULL,
    list_id bigint NOT NULL,
    revision bigint NOT NULL,
    format contacts.vcard_format NOT NULL,
    vcard jsonb NOT NULL,

    CONSTRAINT pk_contacts PRIMARY KEY (user_id, user_type, contact_id),
    CONSTRAINT fk_contacts_user_id_list_id_lists FOREIGN KEY (user_id, user_type, list_id)
        REFERENCES contacts.lists ON DELETE RESTRICT,
    CONSTRAINT check_contact_id CHECK (
        contact_id > 0
    ),
    CONSTRAINT check_revision CHECK (
        revision > 0
    )
);

CREATE INDEX i_contacts_user_id_user_type
    ON contacts.contacts (user_id, user_type);

CREATE INDEX i_contacts_user_id_user_type_list_id
    ON contacts.contacts (user_id, user_type, list_id);

CREATE TABLE contacts.emails (
    user_id bigint NOT NULL,
    user_type contacts.user_type NOT NULL,
    email_id bigint NOT NULL,
    contact_id bigint NOT NULL,
    revision bigint NOT NULL,
    email text NOT NULL,
    type text[],
    label text,

    CONSTRAINT pk_emails PRIMARY KEY (user_id, user_type, email_id),
    CONSTRAINT fk_emails_user_id_user_type_contact_id
        FOREIGN KEY (user_id, user_type, contact_id) REFERENCES contacts.contacts ON DELETE RESTRICT,
    CONSTRAINT check_revision CHECK (
        revision > 0
    ),
    CONSTRAINT check_email CHECK (
        email <> ''
    ),
    CONSTRAINT check_type CHECK (
        type IS NULL OR array_ndims(type) = 1
    )
);

CREATE UNIQUE INDEX uk_emails_user_id_user_type_email
    ON contacts.emails (user_id, user_type, email);

CREATE INDEX i_emails_user_id_user_type
    ON contacts.emails (user_id, user_type);

CREATE INDEX i_emails_user_id_user_type_contact_id
    ON contacts.emails (user_id, user_type, contact_id);

CREATE TABLE contacts.contacts_tags (
    user_id bigint NOT NULL,
    user_type contacts.user_type NOT NULL,
    contact_id bigint NOT NULL,
    tag_id bigint NOT NULL,
    revision bigint NOT NULL,

    CONSTRAINT pk_contacts_tags PRIMARY KEY (user_id, user_type, contact_id, tag_id),
    CONSTRAINT fk_contacts_tags_user_id_user_type_contact_id_contacts FOREIGN KEY (user_id, user_type, contact_id)
        REFERENCES contacts.contacts ON DELETE RESTRICT,
    CONSTRAINT fk_contacts_tags_user_id_user_type_tag_id_tags FOREIGN KEY (user_id, user_type, tag_id)
        REFERENCES contacts.tags ON DELETE RESTRICT,
    CONSTRAINT check_revision CHECK (
        revision > 0
    )
);

CREATE INDEX i_contacts_tags_user_id_user_type_contact_id
    ON contacts.contacts_tags (user_id, user_type, contact_id);

CREATE INDEX i_contacts_tags_user_id_user_type_tag_id
    ON contacts.contacts_tags (user_id, user_type, tag_id);

CREATE TABLE contacts.emails_tags (
    user_id bigint NOT NULL,
    user_type contacts.user_type NOT NULL,
    contact_id bigint NOT NULL,
    email_id bigint NOT NULL,
    tag_id bigint NOT NULL,
    revision bigint NOT NULL,

    CONSTRAINT pk_emails_tags PRIMARY KEY (user_id, user_type, email_id, tag_id),
    CONSTRAINT fk_emails_tags_user_id_user_type_email_id_emails
        FOREIGN KEY (user_id, user_type, email_id) REFERENCES contacts.emails ON DELETE RESTRICT,
    CONSTRAINT fk_emails_tags_user_id_user_type_tag_id_tags
        FOREIGN KEY (user_id, user_type, tag_id) REFERENCES contacts.tags ON DELETE RESTRICT,
    CONSTRAINT fk_emails_tags_user_id_user_type_contact_id_contacts_tags
        FOREIGN KEY (user_id, user_type, contact_id, tag_id)
        REFERENCES contacts.contacts_tags ON DELETE RESTRICT,
    CONSTRAINT check_revision CHECK (
        revision > 0
    )
);

CREATE INDEX i_emails_tags_user_id_user_type_email_id
    ON contacts.emails_tags (user_id, user_type, email_id);

CREATE INDEX i_emails_tags_user_id_user_type_tag_id
    ON contacts.emails_tags (user_id, user_type, tag_id);

CREATE INDEX i_emails_tags_user_id_user_type_contact_id_tag_id
    ON contacts.emails_tags (user_id, user_type, contact_id, tag_id);

CREATE TABLE contacts.shared_lists (
    user_id bigint NOT NULL,
    user_type contacts.user_type NOT NULL,
    list_id bigint NOT NULL,
    client_user_id bigint NOT NULL,
    client_user_type contacts.user_type NOT NULL,
    revision bigint NOT NULL,

    CONSTRAINT pk_shared_lists PRIMARY KEY (user_id, user_type, list_id, client_user_id, client_user_type),
    CONSTRAINT fk_shared_lists_user_id_list_id_lists FOREIGN KEY (user_id, user_type, list_id)
        REFERENCES contacts.lists ON DELETE RESTRICT,
    CONSTRAINT check_share_for_itself CHECK (
        user_id <> client_user_id OR user_type <> client_user_type
    ),
    CONSTRAINT check_client_user_id CHECK (
        client_user_id > 0
    ),
    CONSTRAINT check_revision CHECK (
        revision > 0
    )
);

CREATE INDEX i_shared_lists_user_id_user_type
    ON contacts.shared_lists (user_id, user_type);

CREATE INDEX i_shared_lists_user_id_user_type_list_id
    ON contacts.shared_lists (user_id, user_type, list_id);

CREATE TYPE contacts.change_type AS ENUM (
    'create_user',
    'delete_user',
    'create_list',
    'delete_list',
    'update_list',
    'share_list',
    'revoke_list',
    'create_contacts',
    'delete_contacts',
    'update_contacts',
    'create_tag',
    'delete_tag',
    'update_tag',
    'tag_contacts',
    'untag_contacts',
    'create_emails',
    'delete_emails',
    'update_emails',
    'tag_emails',
    'untag_emails',
    'copy_abook'
);

CREATE SEQUENCE contacts.change_ids;

CREATE TABLE contacts.change_log (
    change_id bigint NOT NULL DEFAULT nextval('contacts.change_ids'),
    user_id bigint NOT NULL,
    user_type contacts.user_type NOT NULL,
    revision bigint NOT NULL,
    type contacts.change_type NOT NULL,
    change_date timestamp with time zone NOT NULL DEFAULT current_timestamp,
    arguments jsonb NOT NULL,
    changed jsonb NOT NULL,
    x_request_id text NOT NULL,
    revert_change_id bigint,
    db_user text DEFAULT CURRENT_USER
) PARTITION BY RANGE (change_date);

CREATE TABLE contacts.change_log_templ (
    LIKE contacts.change_log INCLUDING DEFAULTS
);

ALTER TABLE contacts.change_log_templ
    ADD CONSTRAINT pk_change_log_templ PRIMARY KEY (change_id);

CREATE INDEX i_user_changes_by_revision_templ
    ON contacts.change_log_templ (user_id, user_type, revision);

CREATE INDEX i_inverse_changes_templ
    ON contacts.change_log_templ (revert_change_id);

SELECT create_parent(
    p_parent_table := 'contacts.change_log',
    p_template_table := 'contacts.change_log_templ',
    p_control := 'change_date',
    p_type := 'native',
    p_interval := 'daily',
    p_premake := 5,
    p_jobmon := false
);

UPDATE part_config
   SET retention = '90 days',
       retention_keep_table = false,
       retention_keep_index = false
 WHERE parent_table = 'contacts.change_log';

CREATE TABLE contacts.equivalent_revisions (
    user_id bigint NOT NULL,
    user_type contacts.user_type NOT NULL,
    earlier_revision bigint NOT NULL,
    later_revision bigint NOT NULL,
    add_date timestamp with time zone NOT NULL DEFAULT current_timestamp,

    CONSTRAINT check_revision CHECK (
        earlier_revision < later_revision
    )
) PARTITION BY RANGE (add_date);

CREATE TABLE contacts.equivalent_revisions_templ (
    LIKE contacts.equivalent_revisions INCLUDING DEFAULTS
);

ALTER TABLE contacts.equivalent_revisions_templ
    ADD CONSTRAINT pk_equivalent_revisions_templ
        PRIMARY KEY (user_id, user_type, earlier_revision, later_revision);

CREATE INDEX i_user_equivalent_revisions_templ_by_earlier
    ON contacts.equivalent_revisions_templ (user_id, user_type, earlier_revision);

CREATE INDEX i_user_equivalent_revisions_templ_by_later
    ON contacts.equivalent_revisions_templ (user_id, user_type, later_revision);

SELECT create_parent(
    p_parent_table := 'contacts.equivalent_revisions',
    p_template_table := 'contacts.equivalent_revisions_templ',
    p_control := 'add_date',
    p_type := 'native',
    p_interval := 'daily',
    p_premake := 5,
    p_jobmon := false
);

UPDATE part_config
   SET retention = '14 days',
       retention_keep_table = false,
       retention_keep_index = false
 WHERE parent_table = 'contacts.equivalent_revisions';
