ALTER TABLE mail.serials
    ADD COLUMN next_backup_id integer;

-- create backup tables
CREATE SCHEMA backup;

CREATE TYPE backup.state AS ENUM (
    'new',
    'in_progress',
    'complete',
    'inactive',
    'error'
);

CREATE TABLE backup.backups (
    uid                 bigint NOT NULL,
    backup_id           integer NOT NULL,

    version             integer NOT NULL DEFAULT 1,
    message_count       integer,
    state               backup.state NOT NULL DEFAULT 'new',
    revision            bigint,
    created             timestamptz NOT NULL DEFAULT current_timestamp,
    updated             timestamptz NOT NULL DEFAULT current_timestamp,
    notice              text,

    CONSTRAINT pk_backups PRIMARY KEY (uid, backup_id),
    CONSTRAINT fk_backups_mail_users FOREIGN KEY (uid) REFERENCES mail.users ON DELETE RESTRICT,

    CONSTRAINT check_message_count CHECK (
        message_count IS NULL OR message_count >= 0
    )
);

COMMENT ON COLUMN backup.backups.message_count
    IS 'the number of backuped e-mails';

CREATE INDEX i_backups_state_updated
    ON backup.backups (state, updated);

CREATE INDEX i_backups_uid_state
    ON backup.backups (uid, state);

CREATE UNIQUE INDEX uk_backups_uid_state
    ON backup.backups (uid)
 WHERE state = 'complete';


CREATE TABLE backup.restores (
    uid                 bigint NOT NULL,
    backup_id           integer NOT NULL,
    created             timestamptz NOT NULL DEFAULT current_timestamp,
    updated             timestamptz NOT NULL DEFAULT current_timestamp,
    restore_count       integer,
    state               backup.state NOT NULL DEFAULT 'new',
    notice              text,

    CONSTRAINT pk_restores PRIMARY KEY (uid, backup_id, created),
    CONSTRAINT fk_restores_backups FOREIGN KEY (uid, backup_id) REFERENCES backup.backups ON DELETE RESTRICT,

    CONSTRAINT check_message_count CHECK (
        restore_count IS NULL OR restore_count >= 0
    )
);

COMMENT ON TABLE backup.restores
    IS 'information about backup restoring, '
       'the backup can be restored multiple times';

COMMENT ON COLUMN backup.restores.restore_count
    IS 'the number of emails that need to be restored';

CREATE INDEX i_restores_uid_state
    ON backup.restores (uid, state);


CREATE TABLE backup.folders (
    uid                 bigint NOT NULL,
    backup_id           integer NOT NULL,

    fid                 integer NOT NULL,
    type                mail.folder_types NOT NULL,
    name                text NOT NULL,
    parent_id           integer,

    CONSTRAINT pk_folders PRIMARY KEY (uid, backup_id, fid),
    CONSTRAINT fk_folders_backups FOREIGN KEY (uid, backup_id) REFERENCES backup.backups ON DELETE RESTRICT
);

COMMENT ON TABLE backup.folders
     IS 'copy of the mail.folders at the time of the backup';


CREATE TABLE backup.box (
    uid                 bigint NOT NULL,
    backup_id           integer NOT NULL,
    mid                 bigint NOT NULL,

    st_id               text NOT NULL,
    fid                 integer NOT NULL,
    tab                 mail.tab_types,
    received_date       timestamptz NOT NULL,

    CONSTRAINT pk_box PRIMARY KEY (uid, backup_id, mid),
    CONSTRAINT fk_box_folders FOREIGN KEY (uid, backup_id, fid) REFERENCES backup.folders ON DELETE RESTRICT
);

CREATE INDEX i_box_uid_backup_id_received_date
    ON backup.box (uid, backup_id, received_date);

CREATE INDEX i_box_uid_backup_id_fid
    ON backup.box (uid, backup_id, fid);

CREATE INDEX if_box_hashtext_st_id
    ON backup.box (hashtext(st_id));


CREATE TABLE backup.folders_to_backup (
    uid                 bigint NOT NULL,
    fid                 integer NOT NULL, 

    CONSTRAINT pk_folders_to_backup PRIMARY KEY (uid, fid),
    CONSTRAINT fk_folders_mail_folders FOREIGN KEY (uid, fid) REFERENCES mail.folders ON DELETE CASCADE
);

COMMENT ON TABLE backup.folders_to_backup
     IS 'for specifying the folders to be backed up';
