# coding: utf-8

import re

from .types import MailLabelDef

FLAGS_MAP = {
    'recent': 32,
    'seen': 1,
}

FALSE_STR_VALUES = frozenset(('false', 'no'))
TRUE_STR_VALUES = frozenset(('true', 'yes'))


def cast_bool(value):
    if value.lower() in TRUE_STR_VALUES:
        return True
    elif value.lower() in FALSE_STR_VALUES:
        return False
    else:
        raise ValueError('Bad str value for bool: %s' % value)


def split_list(s_seq):
    return [v for v in [v.strip() for v in s_seq.split(",")] if v]


def lstr2def(lstr):
    return MailLabelDef(**dict(zip(['type', 'name'], lstr.split(':'))))


def labels_from_str(s_seq):
    return [lstr2def(lstr) for lstr in split_list(s_seq)]


def thread_labels_from_str(s_seq):
    res = {}
    for ld_with_count in split_list(s_seq):
        lstr, count = ld_with_count.split('=')
        res[lstr2def(lstr.strip())] = int(count.strip())
    return res


def flags2mask(f_seq):
    if not f_seq:
        return 0
    return sum([FLAGS_MAP[name] for name in set(f_seq)])


def mask2flags(f_mask):
    f_mask = int(f_mask, 2)
    res = []
    for name, bit in FLAGS_MAP.items():
        if f_mask & bit == bit:
            res.append(name)
    return set(res)


VALUES_PATTERN = re.compile(
    r'^\s*(?:'
    r'(?P<list>\$?\w+(?:\s*,\s*\$?\w+)*)'
    r'|(?P<range>\$?\w*\[\d+:\d+\]\w*)'
    r')\s*$'
)

RANGE_PATTERN = re.compile(
    r'^(?P<variable>\$)?'
    r'(?P<prefix>\w*)\['
    r'(?P<lower>\d+):(?P<upper>\d+)'
    r'\](?P<postfix>\w*)$'
)


def parse_values(text):
    """ parse ranges and values

    >>> parse_values('$1, $2')
    ['$1', '$2']
    >>> parse_values('$[1:3]')
    ['$1', '$2', '$3']
    >>> parse_values('x[1:3]')
    ['x1', 'x2', 'x3']
    >>> parse_values('a[1:5]b')
    ['a1b', 'a2b', 'a3b', 'a4b', 'a5b']
    >>> parse_values('$1, $5, $3')
    ['$1', '$5', '$3']
    """
    match = VALUES_PATTERN.search(text)
    if not match:
        raise ValueError('Syntax error in string "%s"' % text)
    match.group('list')
    if match.group('list'):
        return split_list(match.group('list'))
    elif match.group('range'):
        range_match = RANGE_PATTERN.search(match.group('range'))
        if not range_match:
            raise ValueError('Syntax error in string "%s" at substring "%s"'
                             % (text, match.group('range')))
        variable, prefix, lower, upper, postfix = range_match.groups()
        return ['%s%s%d%s' % (variable or '', prefix or '', x, postfix or '')
                for x in range(int(lower), int(upper) + 1)]
    else:
        raise ValueError('Syntax error in "%s"' % text)


def attach_from_str(astr):
    items = astr.split(':')
    res = dict(zip(['hid', 'type', 'filename', 'size'], items))
    for k, v in res.items():
        if v == 'null':
            res[k] = None
    res['size'] = int(res['size'])
    return res


def attaches_from_str(s_seq):
    if s_seq == 'NULL':
        return None
    return [attach_from_str(astr) for astr in split_list(s_seq)]


def mime_part_from_str(astr):
    items = astr.split(':')
    res = dict(zip([
        'hid', 'content_type', 'content_subtype', 'boundary', 'name',
        'charset', 'encoding', 'content_disposition', 'filename', 'cid',
        'offset_begin', 'offset_end'
        ], items))
    for k, v in res.items():
        if v == 'null':
            res[k] = None
    res['offset_begin'] = int(res['offset_begin'])
    res['offset_end'] = int(res['offset_end'])
    return res


def mime_from_str(s_seq):
    if s_seq == 'NULL':
        return None
    return [mime_part_from_str(astr) for astr in split_list(s_seq)]


def recipient_from_str(astr):
    items = astr.split(':')
    res = dict(zip(['type', 'name', 'email'], items))
    for k, v in res.items():
        if v == 'null':
            res[k] = None
    return res


def recipients_from_str(s_seq):
    return [recipient_from_str(astr) for astr in split_list(s_seq)]


def mailish_from_str(mailish_str):
    coords = mailish_str.split("%")
    return {"imap_id": int(coords[0]), "imap_time": coords[1]}
