# coding: utf-8

import six
import re
from itertools import repeat
import logging
from abc import ABCMeta, abstractproperty

import psycopg2.extensions as PGE
from psycopg2.extras import Json
from psycopg2 import IntegrityError, InternalError, DataError, DatabaseError
from mail.pypg.pypg.query_conf import strip_q

from .types import (
    Folder, Label, OperationResult,
    StoreResult, SyncResult, RestoreResult,
    CreateFolderResult, RegisterUserResult,
    ChangeType, SharedFolderSubscription,
    QuickSaveResult, UnsubscribeTask,
    ContactsUserType, ImapIdResult, Tab,
    CollectorIdResult,
    BackupSettings,
    StoreDeletedResult,
    CreateReplyLaterStickerResult,
)

# pylint: disable=R0913

log = logging.getLogger(__name__)


class OperationError(Exception):
    pass


def match_pg_error(exc, error_matches):
    l_error = exc.pgerror.lower() if exc.pgerror else ''

    def it_matches(match):
        if isinstance(match, (six.binary_type, six.text_type)):
            return match in l_error or match in exc.pgerror
        else:
            return match(exc)

    for match, ErrorClass in error_matches:
        if it_matches(match):
            raise ErrorClass(exc.pgerror)


class ErrorTextMatcher(object):
    def __init__(self, pattern):
        self._reg = re.compile(pattern, re.UNICODE | re.IGNORECASE)

    def __call__(self, exc):
        return self._reg.search(exc.pgerror) is not None


class BaseOperation(object):
    __metaclass__ = ABCMeta
    __global_callbacks = []
    db_schema = 'code'
    result_type = dict

    def __init__(self, conn):
        assert hasattr(conn, 'cursor'), "got strange conn: {0}".format(conn)
        self.conn = conn
        self.cur = None
        self.args = None
        self._xid = None

    def __repr__(self):
        return '{name}(conn:{conn})'.format(
            name=self.__class__.__name__,
            conn=self.conn.desc()
        )

    @abstractproperty
    @classmethod
    def db_func_name(cls):
        pass

    def _init_cursor(self):
        if self.cur is not None:
            raise RuntimeError("has not closed cursor: %r" % self.cur)

        self.cur = self.conn.cursor()

        if self.conn.async_ and not self.in_transaction():
            self.cur.execute('BEGIN')
            self.wait()

    def _make_query(self, args):
        return "SELECT txid_current(), * FROM %s.%s(%s)" % (
            self.db_schema, self.db_func_name,
            ", ".join(repeat("%s", len(args)))
        )

    def _execute_query(self, query, args):
        self._init_cursor()

        try:
            self.cur.execute(query, args)
        except Exception as exc:
            self._on_error(exc)
            # reraise exception if _on_error don't did it
            raise
        self.args = args

    def _call(self, *args):
        query = self._make_query(args)
        self._execute_query(query, args)
        return self

    def _fetch_as(self, Result):
        if self.is_waiting():
            self.conn.wait()
        cols = [c.name.lower() for c in self.cur.description]
        result_as_dicts = (dict(zip(cols, r)) for r in self.cur.fetchall())
        result = []
        for d in result_as_dicts:
            self._xid = d.pop('txid_current')
            result.append(Result(**d))
        return result

    @property
    def xid(self):
        if self._xid is None:
            raise RuntimeError(
                'xid not defined, probably {} not finished'.format(self))
        return self._xid

    @property
    def pid(self):
        return self.conn.get_backend_pid()

    def wait(self):
        self.conn.wait()
        return self

    def wait_some(self, tm):
        self.conn.wait_some(tm)
        return self

    def is_waiting(self):
        if self.conn.async_:
            return self.conn.isexecuting()
        return False

    def in_transaction(self):
        return self.conn.get_transaction_status() in (
            PGE.TRANSACTION_STATUS_ACTIVE,
            PGE.TRANSACTION_STATUS_INTRANS
        )

    def add_result_callback(self, func):
        if hasattr(self, 'result_callback'):
            raise RuntimeError(
                'result callback already defined: {0}'.format(
                    self.result_callback)
            )
        self.result_callback = func
        return self

    @classmethod
    def add_callback(cls, c):
        cls.__global_callbacks.append(c)

    def _list_callbacks(self):
        return self.__global_callbacks

    def fetch(self):
        if self.is_waiting():
            self.conn.wait()
        if self.result_type is not None and not hasattr(self, '_result'):
            self._result = self._fetch_as(self.result_type)  # pylint: disable=E1101,W0201
            if hasattr(self, 'result_callback'):
                self.result_callback(self._result)
        for c in self._list_callbacks():
            c(self)
        return self

    @property
    def result(self):
        try:
            return self._result
        except AttributeError:
            if hasattr(self, 'result_type'):
                raise RuntimeError(
                    "operation {0} not finished".format(self))
            raise RuntimeError(
                "operation {0} don't support results".format(self))

    def _on_error(self, exc):
        for ErrorClass, errors_key in [
                (IntegrityError, 'integrity_errors'),
                (InternalError, 'internal_errors'),
                (DataError, 'data_errors'),
                (DatabaseError, 'database_errors')]:
            if isinstance(exc, ErrorClass) and hasattr(self, errors_key):
                match_pg_error(exc, getattr(self, errors_key))

    def _do(self, what):
        self.fetch()
        if not self.conn.async_ and self.conn.autocommit:
            self.wait()
            self.cur.close()
            self.cur = None
            return self
        what = what.upper()
        if self.is_waiting():
            self.conn.wait()

        assert what in ('COMMIT', 'ROLLBACK'),\
            'unsupported action: {0}'.format(what)
        if self.cur is None:
            raise RuntimeError("Can't %r not cursor found: %r" % (what, self))

        self.cur.execute(what)
        self.wait()
        self.cur.close()
        self.cur = None
        return self

    def do(self, what):
        try:
            return self._do(what)
        except Exception as exc:
            self._on_error(exc)
            raise

    def commit(self):
        return self.do('COMMIT')

    def rollback(self):
        return self.do('ROLLBACK')

    def close(self, force=False):
        if not force and self.is_waiting():
            self.conn.wait()
        if self.cur is not None:
            self.cur.close()

    @property
    def simple_result(self):
        return self.result[0][self.db_func_name]


class PositionalQueryMixin(object):
    def _make_query(self, args):
        return "SELECT txid_current(), * FROM %s.%s(%s)" % (
            self.db_schema, self.db_func_name,
            ", ".join(repeat("%s", len(args)))
        )

    def _call(self, *args):
        query = self._make_query(args)
        self._execute_query(query, args)
        return self


class NamedQueryMixin(object):
    '''
    Build query for procedure from __call__ **kwargs
    '''

    @staticmethod
    def _format_args(arg_names):
        return ', '.join(
            '{k} => %({k})s'.format(k=k)
            for k in arg_names
        )

    def _make_named_query(self, arg_names):
        return 'SELECT txid_current(), * FROM {schema}.{function}({args})'.format(
            schema=self.db_schema,
            function=self.db_func_name,
            args=self._format_args(arg_names)
        )

    def __call__(self, **kwargs):
        self._execute_query(
            self._make_named_query(kwargs.keys()),
            kwargs)
        return self


class UserOperation(BaseOperation, PositionalQueryMixin):
    __metaclass__ = ABCMeta
    __global_callbacks = []

    def __init__(self, conn, uid, request_info=None):
        super(UserOperation, self).__init__(conn)
        if six.PY2:
            assert isinstance(uid, (int, long)), "got strange uid:{0}".format(uid)  # noqa
        else:
            assert isinstance(uid, int), "got strange uid:{0}".format(uid)
        self.uid = uid
        self.request_info = request_info

    def __repr__(self):
        return '{name}(uid:{uid}, conn:{conn})'.format(
            name=self.__class__.__name__,
            uid=self.uid,
            conn=self.conn.desc()
        )

    def _list_callbacks(self):
        return super(UserOperation, self)._list_callbacks() + self.__global_callbacks

    @classmethod
    def add_callback(cls, c):
        cls.__global_callbacks.append(c)


class UserIsNotHere(OperationError):
    pass


def handle_user_is_not_here(cls):
    orig_errors = getattr(cls, 'database_errors', tuple())
    cls.database_errors = orig_errors + (
        (lambda exc: exc.pgcode == 'MDBA1', UserIsNotHere),
    )
    return cls


class UIDOnlyOperation(UserOperation):
    def __call__(self):
        return self._call(self.uid)


class UserAlreadyExists(OperationError):
    pass


class RegistrationInProgress(OperationError):
    pass


class HidMisMatch(OperationError):
    pass


class StidMismatch(OperationError):
    pass


class Init(UserOperation):
    db_func_name = 'register_user'
    result_type = RegisterUserResult
    change_type = ChangeType.register

    internal_errors = (
        (ErrorTextMatcher(r'can\'t register new user.*it already exists'),
         UserAlreadyExists),
        (ErrorTextMatcher(r'can\'t register new user.*registration already in progress'),
         RegistrationInProgress),
    )

    def __call__(self, need_welcomes=False, need_filters=False, country=None, lang=None):
        return self._call(self.uid, country, lang, need_welcomes, self.request_info, need_filters)


class FolderAlreadyExists(OperationError):
    pass


class FolderTypeAlreadyExists(OperationError):
    pass


class FolderTypeChangeDefault(OperationError):
    pass


class FolderWithSameFidAlreadyExists(OperationError):
    pass


class FolderWithEmptyName(OperationError):
    pass


@handle_user_is_not_here
class CreateFolder(UserOperation):
    db_func_name = 'create_folder'
    result_type = CreateFolderResult
    integrity_errors = (
        ('uk_folders_uid_name_parent_fid', FolderAlreadyExists),
        ('uk_folders_uid_type_is_unique_type', FolderTypeAlreadyExists),
        ('pk_folders', FolderWithSameFidAlreadyExists),
        ('check_non_empty_name', FolderWithEmptyName)
    )
    change_type = ChangeType.folder_create

    def __call__(
            self, name, parent_fid=None,
            type=Folder.USER):  # pylint: disable=W0622
        return self._call(
            self.uid, name, parent_fid, type, self.request_info
        )

    @property
    def fid(self):
        return self.result[0].fid


@handle_user_is_not_here
class GetOrCreateFolder(UserOperation):
    db_func_name = 'get_or_create_folder'
    result_type = CreateFolderResult
    integrity_errors = (
        ('uk_folders_uid_type_is_unique_type', FolderTypeAlreadyExists),
        ('pk_folders', FolderWithSameFidAlreadyExists),
        ('check_non_empty_name', FolderWithEmptyName)
    )
    change_type = ChangeType.folder_create

    def __call__(
            self, name, parent_fid=None,
            type=Folder.USER):  # pylint: disable=W0622
        return self._call(
            self.uid, name, parent_fid, type, self.request_info
        )

    @property
    def fid(self):
        return self.result[0].fid


class FolderDeleteDefault(OperationError):
    pass


class DeleteFolder(UserOperation):
    db_func_name = 'delete_folder'
    change_type = ChangeType.folder_delete
    internal_errors = (
        (ErrorTextMatcher('can\'t remove folder.*it is default'),
         FolderDeleteDefault),
    )

    def __call__(self, fid):
        return self._call(self.uid, fid, self.request_info)


class LabelWithSameLidAlreadyExists(OperationError):
    pass


class LabelWithNameAndTypeLidAlreadyExists(OperationError):
    pass


class LabelWithEmptyName(OperationError):
    pass


@handle_user_is_not_here
class CreateLabel(UserOperation):
    db_func_name = 'create_label'
    result_type = Label
    integrity_errors = (
        ('pk_labels', LabelWithSameLidAlreadyExists),
        ('uk_labels_uid_type_name', LabelWithNameAndTypeLidAlreadyExists),
        ('check_non_empty_name', LabelWithEmptyName)
    )
    change_type = ChangeType.label_create

    def __call__(self, name, type, color):
        return self._call(self.uid, name, type, color, self.request_info)


@handle_user_is_not_here
class ResolveLabels(UserOperation):
    db_func_name = 'resolve_labels'
    result_type = Label
    integrity_errors = (
        ('pk_labels', LabelWithSameLidAlreadyExists),
    )
    change_type = ChangeType.label_create

    def __call__(self, labels):
        return self._call(self.uid, labels, self.request_info)


class UpdateLabel(UserOperation):
    db_func_name = 'update_label'
    result_type = Label
    change_type = ChangeType.label_modify

    def __call__(self, lid, name, color):
        return self._call(self.uid, lid, name, color, self.request_info)


class InvalidTIDsError(OperationError):
    pass


class JoinThreads(UserOperation):
    db_func_name = 'join_threads'
    internal_errors = (
        ("can't join threads", InvalidTIDsError,),
    )
    change_type = ChangeType.threads_join

    def __call__(self, tid, join_tids):
        return self._call(self.uid, tid, join_tids, self.request_info)


class MailAlreadyExsits(OperationError):
    pass


class ConcurrentThreadInsert(OperationError):
    pass


class StidWithMulca2Prefix(OperationError):
    pass


class EmptyMime(OperationError):
    pass


class MailishAlreadyExists(OperationError):
    pass


@handle_user_is_not_here
class StoreMessage(UserOperation):
    db_func_name = 'store_message_with_tab'
    result_type = StoreResult
    integrity_errors = (
        ('pk_mailbox', MailAlreadyExsits),
        ('pk_threads', ConcurrentThreadInsert),
        ('messages_st_id_without_mulca_2', StidWithMulca2Prefix),
        ('check_plain_not_empty_mime', EmptyMime),
        ('pk_mailish_messages', MailishAlreadyExists)
    )
    change_type = ChangeType.store

    def __call__(
            self, coords,
            headers, recipients,
            attaches, mime, lids, threads_meta, mailish_coords=None, quiet=None):
        return self._call(
            self.uid, coords, headers, recipients, attaches,
            lids, threads_meta, self.request_info, mime, mailish_coords, quiet
        )


class SyncMessageError(OperationError):
    pass


class UserDontHaveSubscribedFolder(SyncMessageError):
    pass


@handle_user_is_not_here
class SyncMessage(UserOperation):
    db_func_name = 'sync_message'
    result_type = SyncResult
    # inherit store_message errors
    integrity_errors = StoreMessage.integrity_errors
    internal_errors = (
        ('do not have subscribed_folders from owner',
         UserDontHaveSubscribedFolder),
    )
    change_type = ChangeType.sync_store

    def __call__(
            self, owner_coords, sync_coords,
            headers, recipients,
            attaches, mime, lids, threads, quiet=None):
        return self._call(
            self.uid, owner_coords, sync_coords,
            headers, recipients,
            attaches, lids, threads,
            self.request_info, mime, quiet
        )


class QuickSaveMessage(UserOperation):
    db_func_name = 'quick_save_message'
    integrity_errors = (
        ('pk_mailbox', MailAlreadyExsits),
        ('pk_threads', ConcurrentThreadInsert),
        ('check_plain_not_empty_mime', EmptyMime),
    )
    change_type = ChangeType.quick_save
    result_type = QuickSaveResult

    def __call__(self, mid, coords, headers, recipients, attaches, mime):
        return self._call(
            self.uid, mid, coords, headers, recipients, attaches,
            self.request_info, mime
        )


@handle_user_is_not_here
class UpdateMessages(UserOperation):
    db_func_name = 'update_messages'
    result_type = OperationResult
    change_type = ChangeType.update

    def __call__(
            self, mids, seen, recent, deleted,
            lids_add=None, lids_del=None):
        return self._call(
            self.uid, mids, seen, recent, deleted,
            lids_add or [], lids_del or [],
            self.request_info
        )


@handle_user_is_not_here
class MoveMessages(UserOperation):
    db_func_name = 'move_messages'
    result_type = OperationResult
    change_type = ChangeType.move

    def __call__(self, mids, new_fid, new_tab):
        return self._call(self.uid, mids, new_fid, new_tab, self.request_info)


@handle_user_is_not_here
class CopyMessages(UserOperation):
    db_func_name = 'copy_messages'
    result_type = OperationResult
    change_type = ChangeType.copy

    def __call__(self, mids, dst_fid, dst_tab=None):
        return self._call(self.uid, mids, dst_fid, self.request_info, dst_tab)


@handle_user_is_not_here
class DeleteMessages(UserOperation):
    db_func_name = 'delete_messages'
    result_type = OperationResult
    change_type = ChangeType.delete

    def __call__(self, mids):
        return self._call(self.uid, mids, self.request_info)


@handle_user_is_not_here
class ResetFresh(UserOperation):
    db_func_name = 'reset_fresh'
    change_type = ChangeType.fresh_reset

    def __call__(self):
        return self._call(self.uid, self.request_info)


class ResetFolderUnvisited(UserOperation):
    db_func_name = 'reset_folder_unvisited'
    change_type = ChangeType.folder_reset_unvisited

    def __call__(self, fid):
        return self._call(self.uid, fid)


class UpdateFolderDefault(OperationError):
    pass


class UpdateFolder(UserOperation):
    db_func_name = 'update_folder'
    change_type = ChangeType.folder_modify
    internal_errors = (
        (ErrorTextMatcher(r'can\'t update folder.*it is default'),
         UpdateFolderDefault),
    )

    def __call__(self, fid, new_name, new_parent):
        return self._call(
            self.uid, fid, new_name, new_parent, self.request_info)


class UpdateFolderType(UserOperation):
    db_func_name = 'update_folder_type'
    integrity_errors = (
        ('uk_folders_uid_type_is_unique_type', FolderTypeAlreadyExists),
    )
    internal_errors = (
        (ErrorTextMatcher('can\'t update type for folder.*it is default'),
         FolderTypeChangeDefault),
    )
    change_type = ChangeType.folder_modify_type

    def __call__(self, fid, new_type):
        return self._call(self.uid, fid, new_type, self.request_info)


class InvalidFolderFullName(OperationError):
    pass


class IMAPDeleteUnsubscribed(UserOperation):
    db_func_name = 'imap_delete_unsubscribed'
    change_type = ChangeType.imap_delete_unsubscribed

    def __call__(self, full_name):
        return self._call(self.uid, full_name, self.request_info)


class IMAPAddUnsubscribed(UserOperation):
    db_func_name = 'imap_add_unsubscribed'
    integrity_errors = (
        ('check_no_nulls_and_empty_names_in_full_name', InvalidFolderFullName),
        ('violates not-null constraint', InvalidFolderFullName),
    )
    change_type = ChangeType.imap_add_unsubscribed

    def __call__(self, full_name):
        return self._call(self.uid, full_name, self.request_info)


class NonEmptyLabelError(OperationError):
    pass


class DefaultLabelError(OperationError):
    pass


class DeleteLabel(UserOperation):
    db_func_name = 'delete_label'
    change_type = ChangeType.label_delete
    internal_errors = (
        (ErrorTextMatcher(r'can\'t remove label.*marked messages exist'),
         NonEmptyLabelError),
        (ErrorTextMatcher(r'can\'t remove label.*it is default'),
         DefaultLabelError),
    )

    def __call__(self, lid):
        return self._call(self.uid, lid, self.request_info)


class NotInitializedUserError(OperationError):
    pass


class BlacklistAdd(UserOperation):
    db_func_name = 'blacklist_add'
    integrity_errors = (
        ('fk_elist_mail_users', NotInitializedUserError),
    )

    def __call__(self, email):
        return self._call(self.uid, 'black', email)


class BlacklistRemove(UserOperation):
    db_func_name = 'blacklist_remove'

    def __call__(self, email):
        return self._call(self.uid, 'black', [email])


class CreateRule(UserOperation):
    db_func_name = 'create_rule'
    integrity_errors = (
        ('fk_rules_mail_users', NotInitializedUserError),
    )

    def __call__(self, name, enabled, stop, last, acts, conds, old_rule_id, rule_type='user'):
        new_id = None
        return self._call(
            self.uid, name, enabled, stop, last,
            acts, conds, old_rule_id, new_id, rule_type)


class CreateUnsubscribeRule(UserOperation):
    db_func_name = 'create_unsubscribe'
    integrity_errors = (
        ('fk_rules_mail_users', NotInitializedUserError),
    )

    def __call__(self, email, name, message_type, fid):
        params = [email, name, message_type, fid]
        return self._call(
            self.uid, params)


class CreateSystemRules(UserOperation):
    db_func_name = 'create_base_filters'
    integrity_errors = (
        ('fk_rules_mail_users', NotInitializedUserError),
    )

    def __call__(self):
        self.db_schema = 'impl'
        return self._call(self.uid)


class RemoveRule(UserOperation):
    db_func_name = 'rule_remove'

    def __call__(self, rule_id):
        return self._call(self.uid, rule_id)


class RemoveRules(UserOperation):
    db_func_name = 'rules_remove'

    def __call__(self, rule_ids):
        return self._call(self.uid, rule_ids)


class VerifyAction(UserOperation):
    db_func_name = 'verify_action'

    def __call__(self, action_id):
        return self._call(self.uid, action_id)


class EnableRule(UserOperation):
    db_func_name = 'rule_enable'

    def __call__(self, rule_id):
        return self._call(self.uid, rule_id)


class DisableRule(UserOperation):
    db_func_name = 'rule_disable'

    def __call__(self, rule_id):
        return self._call(self.uid, rule_id)


class OrderRules(UserOperation):
    db_func_name = 'order_rules_with_check'

    def __call__(self, rule_ids):
        return self._call(self.uid, rule_ids)


class MessageNotFoundError(OperationError):
    pass


class AttachNotFoundError(OperationError):
    pass


class UnexpectedBehaviorError(OperationError):
    pass


class InvalidFixArgumentError(OperationError):
    pass


class FixUser(UserOperation):
    db_schema = 'util'
    db_func_name = 'fix_user'
    internal_errors = (
        ('unexpected behavior', UnexpectedBehaviorError),
    )
    data_errors = (
        ('invalid input value for enum util.fix', InvalidFixArgumentError),
    )

    def __call__(self, fixes):
        return self._call(self.uid, fixes)


class RecreateImapChains(UserOperation):
    db_schema = 'util'
    db_func_name = 'recreate_imap_chains'

    def __call__(self, fid, chain_size):
        return self._call(self.uid, fid, chain_size)


class FillChangelogForMsearch(UIDOnlyOperation):
    db_func_name = 'fill_changelog_for_msearch'
    db_schema = 'util'
    change_type = ChangeType.reindex


class DeleteFoldersSubtree(UserOperation):
    db_func_name = 'delete_folders_subtree'
    db_schema = 'util'

    def __call__(self, root_fid):
        return self._call(self.uid, root_fid)


class FixNotPositiveLIDs(UIDOnlyOperation):
    db_func_name = 'fix_not_positive_lids'
    db_schema = 'util'


class FixSoLabelTypes(UIDOnlyOperation):
    db_func_name = 'fix_so_label_types'
    db_schema = 'util'


class Migration(UIDOnlyOperation):
    db_schema = 'util'

    @classmethod
    def known_migrations(cls):
        return dict(
            (o.db_func_name, o)
            for o in cls.__subclasses__()  # pylint: disable=E1101
        )

    @classmethod
    def get_migration_by_name(cls, migration):
        return cls.known_migrations()[migration]


class MergeTMIntoMessages(Migration):
    db_func_name = 'merge_tm_into_messages'


class RestoreDeleted(UserOperation):
    db_schema = 'util'
    db_func_name = 'restore_deleted'
    result_type = RestoreResult

    def __call__(self, mids, fid):
        return self._call(self.uid, mids, fid)


class InitializePOP3Folder(UserOperation):
    db_func_name = 'initialize_pop3_folder'
    change_type = ChangeType.pop3_folder_initialization

    def __call__(self, fid):
        return self._call(self.uid, fid, self.request_info)


class POP3FoldersEnable(UserOperation):
    db_func_name = 'pop3_folders_enable'
    change_type = ChangeType.pop3_folders_enable

    def __call__(self, fids):
        return self._call(self.uid, fids, self.request_info)


class POP3FoldersDisable(UserOperation):
    db_func_name = 'pop3_folders_disable'
    change_type = ChangeType.pop3_folders_disable

    def __call__(self, fids):
        return self._call(self.uid, fids, self.request_info)


class POP3Delete(UserOperation):
    db_func_name = 'pop3_delete'
    change_type = ChangeType.pop3_delete
    result_type = OperationResult

    def __call__(self, mids):
        return self._call(self.uid, mids, self.request_info)


class ApplyDataMigrations(UserOperation):
    db_func_name = 'apply_data_migrations'
    db_schema = 'util'

    def __call__(self, target_data_version=None):
        if target_data_version is None:
            return self._call(self.uid)
        return self._call(self.uid, target_data_version)


class DeleteUser(UserOperation):
    db_func_name = 'delete_user'
    change_type = ChangeType.user_delete

    def __call__(self, deleted_date):
        return self._call(self.uid, deleted_date, self.request_info)


class PurgeUser(UIDOnlyOperation):
    db_func_name = 'purge_user'
    db_schema = 'code'
    violating = True


class AddWindat(UserOperation):
    q = u"""INSERT INTO mail.windat_messages
            VALUES (%s, %s, %s, %s, (%s)::mail.mime_part)"""

    integrity_errors = (
        ('check_windat_messages_hid',  HidMisMatch),
        ('check_windat_messages_stid', StidMismatch)
    )

    db_func_name = None
    result_type = None

    def __call__(self, mid, st_id, hid, windat):
        self._execute_query(self.q, [self.uid, mid, st_id, hid, windat])
        return self


class AddFolderToSharedFolders(UserOperation):
    db_func_name = 'add_folder_to_shared_folders'
    change_type = ChangeType.shared_folder_create

    def __call__(self, fid):
        return self._call(self.uid, fid, self.request_info)


class FolderAlreadySubscribedToDifferentSharedFolder(OperationError):
    pass


class SharedFolderAlreadySubscribed(OperationError):
    pass


class AddFolderToSubscribedFolders(UserOperation):
    db_func_name = 'add_folder_to_subscribed_folders'
    integrity_errors = (
        ('pk_subscribed_folders', FolderAlreadySubscribedToDifferentSharedFolder),
        ('uk_subscribed_folders_uid_fid', FolderAlreadySubscribedToDifferentSharedFolder),
        ('uk_subscribed_folders_uid_owner_uid_fid', SharedFolderAlreadySubscribed),
    )
    change_type = ChangeType.subscribed_folder_create

    def __call__(self, fid, owner_uid, owner_fid):
        return self._call(
            self.uid,
            fid,
            owner_uid,
            owner_fid,
            self.request_info
        )


class AddSubscriberToSharedFolders(UserOperation):
    db_func_name = 'add_subscriber_to_shared_folders'
    change_type = ChangeType.shared_folder_subscribe

    def __call__(self, fid, subscriber):
        return self._call(
            self.uid,
            fid,
            subscriber,
            self.request_info
        )


class NotExistedSubscription(OperationError):
    pass


class InvalidSubscriptionStateTransition(OperationError):
    pass


class TransitSubscriptionState(UserOperation):
    db_func_name = 'transit_subscription_state'
    result_type = SharedFolderSubscription

    integrity_errors = []

    internal_errors = [
        ('there are no subscription with', NotExistedSubscription),
        ('there are no known transition from',
         InvalidSubscriptionStateTransition),
    ]

    def __call__(self, subscription_id, action):
        return self._call(
            self.uid, subscription_id, action
        )


class MarkSubscriptionFailed(UserOperation):
    db_func_name = 'mark_subscription_failed'
    result_type = SharedFolderSubscription

    integrity_errors = TransitSubscriptionState.integrity_errors
    internal_errors = TransitSubscriptionState.internal_errors

    def __call__(self, subscription_id, fail_reason):
        return self._call(
            self.uid, subscription_id, fail_reason
        )


class AddToStorageDeleteQueue(UserOperation):
    db_func_name = 'add_to_storage_delete_queue'

    def __call__(self, st_id):
        return self._call(self.uid, st_id)


class PurgeDeletedMessages(UserOperation):
    db_func_name = 'purge_deleted_messages'

    def __call__(self, mids):
        return self._call(self.uid, mids)


@handle_user_is_not_here
class SyncDeleteMessages(UserOperation):
    db_func_name = 'sync_delete_messages'
    result_type = OperationResult
    change_type = ChangeType.sync_delete

    def __call__(self, owner_uid, owner_fid, owner_mids, owner_revision):
        return self._call(
            self.uid,
            owner_uid,
            owner_fid,
            owner_mids,
            owner_revision,
            self.request_info
        )


@handle_user_is_not_here
class SyncJoinThreads(UserOperation):
    db_func_name = 'sync_join_threads'
    result_type = OperationResult
    change_type = ChangeType.sync_threads_join

    def __call__(self, owner_uid, owner_fid, owner_tid, owner_join_tids, owner_revision):
        return self._call(
            self.uid,
            owner_uid,
            owner_fid,
            owner_tid,
            owner_join_tids,
            owner_revision,
            self.request_info
        )


@handle_user_is_not_here
class SyncUpdateMessages(UserOperation):
    db_func_name = 'sync_update_messages'
    result_type = OperationResult
    change_type = ChangeType.sync_delete

    def __call__(
            self, owner_uid, owner_fid, owner_mids, owner_revision,
            seen, recent, deleted,
            lids_add=None, lids_del=None):
        return self._call(
            self.uid,
            owner_uid, owner_fid, owner_mids, owner_revision,
            seen, recent, deleted,
            lids_add or [], lids_del or [],
            self.request_info
        )


class AddingMailishEntryToNonMailishFolder(OperationError):
    pass


class AddMailishEntry(UserOperation):
    db_func_name = 'add_mailish_entry'

    integrity_errors = (
        ('fk_mailish_messages_uid_fid', AddingMailishEntryToNonMailishFolder),
    )

    def __call__(self, fid, mailish_coords, errors=0):
        return self._call(self.uid, fid, mailish_coords, errors)


class EraseMailishSecurityLocks(UserOperation):
    db_func_name = 'erase_mailish_security_locks'

    def __call__(self):
        return self._call(self.uid)


class CreatingMailishFolderWithoutMailishAccount(OperationError):
    pass


class MailishFolderWithEmptyImapPath(OperationError):
    pass


class CreatingMailishFolderFromNonEmptyFolder(OperationError):
    pass


class InitExistingFolderAsMailish(UserOperation):
    db_func_name = 'init_existing_folder_as_mailish'

    integrity_errors = (
        ('fk_mailish_folders_uid', CreatingMailishFolderWithoutMailishAccount),
        ('not_empty_imap_path', MailishFolderWithEmptyImapPath),
    )

    internal_errors = (
        ('non empty folder', CreatingMailishFolderFromNonEmptyFolder,),
    )

    def __call__(self, fid, mailish_folder_info):
        return self._call(self.uid, fid, mailish_folder_info)


class InvalidateMailishAuth(UserOperation):
    db_func_name = 'invalidate_mailish_auth'

    def __call__(self, token_id):
        return self._call(self.uid, token_id)


class SaveMailishAccount(UserOperation):
    db_func_name = 'save_mailish_account'

    def __call__(self, account):
        return self._call(
            self.uid,
            account['email'],
            account['imap_login'],
            account['imap_credentials'],
            account['imap_server'],
            account['imap_port'],
            account['imap_ssl'],
            account['smtp_login'],
            account['smtp_credentials'],
            account['smtp_server'],
            account['smtp_port'],
            account['smtp_ssl'],
            account['token_id'],
            account['auth_type'],
            account['uuid'],
            account['oauth_app'],
            account['last_sync'])


class UpdatingMailishDownloadedRangeWithInvalidValues(OperationError):
    pass


class UpdateMailishDownloadedRange(UserOperation):
    db_func_name = 'update_mailish_downloaded_range'

    integrity_errors = (
        ('valid_downloaded_range', UpdatingMailishDownloadedRangeWithInvalidValues),
    )

    def __call__(self, fid, range_start, range_end):
        return self._call(self.uid, fid, range_start, range_end)


class UpdateMailishFolder(UserOperation):
    db_func_name = 'update_mailish_folder'

    integrity_errors = (
        ('not_empty_imap_path', MailishFolderWithEmptyImapPath),
    )

    def __call__(self, fid, mailish_folder_info):
        return self._call(self.uid, fid, mailish_folder_info)


class DeletingNonEmptyMailishFolderEntry(OperationError):
    pass


class DeleteMailishFolderEntries(UserOperation):
    db_func_name = 'delete_mailish_folder_entries'

    internal_errors = (
        ("deleting non empty folder from mailish.folders", DeletingNonEmptyMailishFolderEntry,),
    )

    def __call__(self, fids):
        return self._call(self.uid, fids)


class DeletingMailishEntryWithNonEmptyMid(OperationError):
    pass


class DeleteMailishEntries(UserOperation):
    db_func_name = 'delete_mailish_entries'

    internal_errors = (
        ("deleting non empty mid entry from mailish.messages", DeletingMailishEntryWithNonEmptyMid,),
    )

    def __call__(self, fid, imap_ids):
        return self._call(self.uid, fid, imap_ids)


class IncrementMailishEntryErrorsCount(UserOperation):
    db_func_name = 'increment_mailish_entry_errors_count'

    integrity_errors = (
        ('fk_mailish_messages_uid_fid', AddingMailishEntryToNonMailishFolder),
    )

    def __call__(self, fid, imap_id, errors=1):
        return self._call(self.uid, fid, imap_id, errors)


class MoveMailishMessages(UserOperation):
    db_func_name = 'move_mailish_messages'

    def __call__(self, src_fid, dst_fid, mailish_move_coords):
        return self._call(self.uid, src_fid, dst_fid, mailish_move_coords)


class UpdateMailishAccountLastSync(UserOperation):
    db_func_name = 'update_mailish_account_last_sync'

    def __call__(self, last_sync):
        return self._call(self.uid, last_sync)


class SubscriptionLimitIsNotPositive(OperationError):
    pass


class GetFreeSubscriptions(BaseOperation, PositionalQueryMixin):
    db_func_name = 'get_free_subscriptions'
    result_type = SharedFolderSubscription

    integrity_errors = (
        ('subscription_limit_not_null_and_positive',
         SubscriptionLimitIsNotPositive,),
    )

    def __call__(self, worker_id, subscription_limit):
        return self._call(worker_id, subscription_limit)


class ReleaseSubscription(UserOperation):
    db_func_name = 'release_subscription'
    result_type = SharedFolderSubscription

    internal_errors = (
        ('there are no subscription with', NotExistedSubscription),
    )

    def __call__(self, subscription_id, worker_id):
        return self._call(
            self.uid, subscription_id, worker_id
        )


class AddDobermanJob(BaseOperation):
    db_func_name = None
    q = strip_q('''
INSERT INTO mail.doberman_jobs
    (worker_id, launch_id, hostname,
     worker_version, assigned,
     heartbeated, timeout)
VALUES
    (%(worker_id)s, %(launch_id)s, %(hostname)s,
     %(worker_version)s, %(assigned)s,
     %(heartbeated)s, %(timeout)s)
    ON CONFLICT DO NOTHING
RETURNING *, txid_current()''')

    def __call__(
            self,
            worker_id,
            launch_id=None,
            worker_version=None,
            hostname=None,
            assigned=None,
            heartbeated=None,
            timeout=None):
        self._execute_query(self.q, locals())
        return self


class FindAJob(BaseOperation, NamedQueryMixin):
    db_func_name = 'find_a_job'


class NonExistentDobermanJob(OperationError):
    pass


class ConfirmTheJob(BaseOperation, NamedQueryMixin):
    db_func_name = 'confirm_the_job'

    internal_errors = [
        (ErrorTextMatcher(r'There are no "[^"]*" worker_id in doberman_jobs'),
         NonExistentDobermanJob),
    ]


class DeleteFolderFromSubscribedFolders(BaseOperation, NamedQueryMixin):
    db_func_name = 'delete_folder_from_subscribed_folders'


class DeleteSharedFolderSubscriptions(UserOperation):
    db_func_name = 'delete_shared_folder_subscriptions'
    result_type = SharedFolderSubscription

    def __call__(self, subscription_ids):
        return self._call(
            self.uid, subscription_ids
        )


class AddUnsubscribeTask(BaseOperation, PositionalQueryMixin):
    db_func_name = 'add_unsubscribe_task'
    result_type = UnsubscribeTask

    integrity_errors = (
        ('fk_unsubscribe_tasks_users', NotInitializedUserError),
    )

    def __call__(
            self,
            task_request_id,
            owner_uid,
            owner_fids,
            subscriber_uid,
            root_subscriber_fid):
        return self._call(
            task_request_id,
            owner_uid,
            owner_fids,
            subscriber_uid,
            root_subscriber_fid
        )


class GetUnsubscribeTasks(BaseOperation, PositionalQueryMixin):
    db_func_name = 'get_unsubscribe_tasks'
    result_type = UnsubscribeTask

    def __call__(self, task_limit, task_ttl):
        return self._call(task_limit, task_ttl)


class DeleteSubscriptionsAndUnsubscribeTask(BaseOperation, PositionalQueryMixin):
    db_func_name = 'delete_subscriptions_with_unsubscribe_task'
    result_type = SharedFolderSubscription

    def __call__(self, task_id):
        return self._call(task_id)


class SetFolderArchivationRule(UserOperation):
    db_func_name = 'set_folder_archivation_rule'
    change_type = ChangeType.set_archivation_rule

    def __call__(self, fid, archive_type, keep_days, max_size=None):
        return self._call(
            self.uid, fid, archive_type, keep_days, None, max_size
        )


class RemoveFolderArchivationRule(UserOperation):
    db_func_name = 'remove_folder_archivation_rule'
    change_type = ChangeType.remove_archivation_rule

    def __call__(self, fid):
        return self._call(
            self.uid, fid
        )


class CreateSettings(UserOperation):
    db_func_name = 'create_settings'

    def __call__(self, settings):
        return self._call(self.uid, settings)


class DeleteSettings(UserOperation):
    db_func_name = 'delete_settings'

    def __call__(self):
        return self._call(self.uid)


class UpdateSettings(UserOperation):
    db_func_name = 'update_settings'

    def __call__(self, settings):
        return self._call(self.uid, settings)


class ContactsUserOperation(BaseOperation, PositionalQueryMixin):
    __metaclass__ = ABCMeta
    __global_callbacks = []

    def __init__(self, conn, user_id, user_type, request_info=None):
        super(ContactsUserOperation, self).__init__(conn)
        if six.PY2:
            assert isinstance(user_id, (int, long)), 'got strange user_id: {0}'.format(user_id)  # noqa
        else:
            assert isinstance(user_id, int), 'got strange user_id: {0}'.format(user_id)
        assert isinstance(user_type, ContactsUserType), 'got strange user_type: {0}'.format(user_type)
        self.user_id = user_id
        self.user_type = user_type
        self.request_info = request_info

    @classmethod
    def by_uid(cls, conn, uid, *args, **kwargs):
        return cls(
            conn=conn,
            user_id=uid,
            user_type=ContactsUserType.passport_user,
            *args,
            **kwargs
        )

    def __repr__(self):
        return '{name}(user_id:{user_id}, user_type:{user_type}, conn:{conn})'.format(
            name=self.__class__.__name__,
            user_id=self.user_id,
            user_type=self.user_type,
            conn=self.conn.desc()
        )

    def _list_callbacks(self):
        return super(ContactsUserOperation, self)._list_callbacks() + self.__global_callbacks

    @classmethod
    def add_callback(cls, c):
        cls.__global_callbacks.append(c)


class CreateContactsUser(ContactsUserOperation):
    db_func_name = 'create_contacts_user'

    def __call__(self, x_request_id):
        return self._call(self.user_id, self.user_type, x_request_id)


class DeleteContactsUser(ContactsUserOperation):
    db_func_name = 'delete_contacts_user'

    def __call__(self, x_request_id):
        return self._call(self.user_id, self.user_type, x_request_id)


class DuplicateContactsList(OperationError):
    pass


class DuplicateUniqueTypeContactsList(OperationError):
    pass


class CreateContactsList(ContactsUserOperation):
    db_func_name = 'create_contacts_list'
    integrity_errors = (
        ('uk_lists_name_and_type_for_user', DuplicateContactsList),
        ('uk_lists_user_id_user_type_unique_type', DuplicateUniqueTypeContactsList),
    )

    def __call__(self, list_name, list_type, x_request_id):
        return self._call(self.user_id, self.user_type, list_name, list_type, x_request_id)


class DeleteDefaultContactsList(OperationError):
    pass


class DeleteContactsList(ContactsUserOperation):
    db_func_name = 'delete_contacts_list'
    internal_errors = (
        ('Delete default list is forbidden', DeleteDefaultContactsList),
    )

    def __call__(self, list_id, x_request_id):
        return self._call(self.user_id, self.user_type, list_id, x_request_id)


class UpdateDefaultContactsList(OperationError):
    pass


class UpdateContactsListWithOutdatedRevision(OperationError):
    pass


class UpdateContactsList(ContactsUserOperation):
    db_func_name = 'update_contacts_list'
    internal_errors = (
        ('Update default list is forbidden', UpdateDefaultContactsList),
        ('Update list with outdated revision', UpdateContactsListWithOutdatedRevision),
    )

    def __call__(self, list_id, list_name, x_request_id, revision=None):
        return self._call(self.user_id, self.user_type, list_id, list_name, x_request_id, revision)


class DuplicateContactsTag(OperationError):
    pass


class CreateContactsTag(ContactsUserOperation):
    db_func_name = 'create_contacts_tag'
    integrity_errors = (
        ('uk_tags_name_and_type_for_user', DuplicateContactsTag),
    )

    def __call__(self, tag_name, tag_type, x_request_id):
        return self._call(self.user_id, self.user_type, tag_name, tag_type, x_request_id)


class DeleteDefaultContactsTag(OperationError):
    pass


class DeleteContactsTag(ContactsUserOperation):
    db_func_name = 'delete_contacts_tag'
    internal_errors = (
        ('Delete default tag is forbidden', DeleteDefaultContactsTag),
    )

    def __call__(self, tag_id, x_request_id):
        return self._call(self.user_id, self.user_type, tag_id, x_request_id)


class UpdateDefaultContactsTag(OperationError):
    pass


class UpdateContactsTagWithOutdatedRevision(OperationError):
    pass


class UpdateContactsTag(ContactsUserOperation):
    db_func_name = 'update_contacts_tag'
    internal_errors = (
        ('Update default tag is forbidden', UpdateDefaultContactsTag),
        ('Update tag with outdated revision', UpdateContactsTagWithOutdatedRevision),
    )

    def __call__(self, tag_id, tag_name, x_request_id, revision=None):
        return self._call(self.user_id, self.user_type, tag_id, tag_name, x_request_id, revision)


class CreateContacts(ContactsUserOperation):
    db_func_name = 'create_contacts'

    def __call__(self, new_contacts, x_request_id):
        return self._call(self.user_id, self.user_type, new_contacts, x_request_id)


class DeleteContacts(ContactsUserOperation):
    db_func_name = 'delete_contacts'

    def __call__(self, contact_ids, x_request_id):
        return self._call(self.user_id, self.user_type, contact_ids, x_request_id)


class UpdateContactsWithOutdatedRevision(OperationError):
    pass


class UpdateContacts(ContactsUserOperation):
    db_func_name = 'update_contacts'
    internal_errors = (
        ('Update contact with outdated revision', UpdateContactsWithOutdatedRevision),
    )

    def __call__(self, contacts, x_request_id, revision=None):
        return self._call(self.user_id, self.user_type, contacts, x_request_id, revision)


class TagContacts(ContactsUserOperation):
    db_func_name = 'tag_contacts'

    def __call__(self, tag_id, contact_ids, x_request_id):
        return self._call(self.user_id, self.user_type, tag_id, contact_ids, x_request_id)


class UntagContactsWithTaggedEmails(OperationError):
    pass


class UntagContacts(ContactsUserOperation):
    db_func_name = 'untag_contacts'
    integrity_errors = (
        ('fk_emails_tags_user_id_user_type_contact_id_contacts_tags', UntagContactsWithTaggedEmails),
    )

    def __call__(self, tag_id, contact_ids, x_request_id):
        return self._call(self.user_id, self.user_type, tag_id, contact_ids, x_request_id)


class UntagContactsCompletely(ContactsUserOperation):
    db_func_name = 'untag_contacts_completely'
    integrity_errors = (
        ('fk_emails_tags_user_id_user_type_contact_id_contacts_tags', UntagContactsWithTaggedEmails),
    )

    def __call__(self, contact_ids, x_request_id):
        return self._call(self.user_id, self.user_type, contact_ids, x_request_id)


class CreateContactsEmails(ContactsUserOperation):
    db_func_name = 'create_contacts_emails'

    def __call__(self, new_emails, x_request_id):
        return self._call(self.user_id, self.user_type, new_emails, x_request_id)


class DeleteContactsEmails(ContactsUserOperation):
    db_func_name = 'delete_contacts_emails'

    def __call__(self, email_ids, x_request_id):
        return self._call(self.user_id, self.user_type, email_ids, x_request_id)


class UpdateContactsEmailsWithOutdatedRevision(OperationError):
    pass


class UpdateContactsEmails(ContactsUserOperation):
    db_func_name = 'update_contacts_emails'
    internal_errors = (
        ('Update email with outdated revision', UpdateContactsEmailsWithOutdatedRevision),
    )

    def __call__(self, emails, x_request_id, revision=None):
        return self._call(self.user_id, self.user_type, emails, x_request_id, revision)


class TagContactsEmailsWithUntaggedContacts(OperationError):
    pass


class TagContactsEmails(ContactsUserOperation):
    db_func_name = 'tag_contacts_emails'
    integrity_errors = (
        ('fk_emails_tags_user_id_user_type_contact_id_contacts_tags', TagContactsEmailsWithUntaggedContacts),
    )

    def __call__(self, tag_id, email_ids, x_request_id):
        return self._call(self.user_id, self.user_type, tag_id, email_ids, x_request_id)


class UntagContactsEmails(ContactsUserOperation):
    db_func_name = 'untag_contacts_emails'

    def __call__(self, tag_id, email_ids, x_request_id):
        return self._call(self.user_id, self.user_type, tag_id, email_ids, x_request_id)


class UntagContactsEmailsCompletely(ContactsUserOperation):
    db_func_name = 'untag_contacts_emails_completely'

    def __call__(self, email_ids, x_request_id):
        return self._call(self.user_id, self.user_type, email_ids, x_request_id)


class ShareContactsList(ContactsUserOperation):
    db_func_name = 'share_contacts_list'

    def __call__(self, list_id, client_user_id, client_user_type, x_request_id):
        return self._call(self.user_id, self.user_type, list_id, client_user_id, client_user_type, x_request_id)


class SubscribeToContactsList(ContactsUserOperation):
    db_func_name = 'subscribe_to_contacts_list'

    def __call__(self, list_id, owner_user_id, owner_user_type, owner_list_id, x_request_id):
        return self._call(self.user_id, self.user_type, list_id, owner_user_id, owner_user_type,
                          owner_list_id, x_request_id)


class RevokeContactsList(ContactsUserOperation):
    db_func_name = 'revoke_contacts_list'

    def __call__(self, list_id, client_user_id, client_user_type, x_request_id):
        return self._call(self.user_id, self.user_type, list_id, client_user_id, client_user_type, x_request_id)


class RevokeSubscribedContactsList(ContactsUserOperation):
    db_func_name = 'revoke_subscribed_contacts_list'

    def __call__(self, list_id, owner_user_id, owner_user_type, owner_list_id, x_request_id):
        return self._call(self.user_id, self.user_type, list_id, owner_user_id, owner_user_type,
                          owner_list_id, x_request_id)


class RevisionNotFound(OperationError):
    pass


class RestoreContacts(ContactsUserOperation):
    db_func_name = 'restore_contacts'
    internal_errors = (
        (ErrorTextMatcher(r'Revision \d+ not found for user'), RevisionNotFound),
    )

    def __call__(self, revision, x_request_id):
        return self._call(self.user_id, self.user_type, revision, x_request_id)


class RegenerateImapId(UserOperation):
    db_func_name = 'regenerate_imap_id'
    result_type = ImapIdResult
    change_type = ChangeType.reindex

    def __call__(self, mid):
        return self._call(self.uid, mid, self.request_info)


class AddContactsDirectoryEvent(ContactsUserOperation):
    db_func_name = 'add_contacts_directory_event'

    def __call__(self, event_id):
        return self._call(self.user_id, self.user_type, event_id)


class CompleteSyncContactsDirectoryEvent(ContactsUserOperation):
    db_func_name = 'complete_sync_contacts_directory_event'

    def __call__(self, synced_revision, synced_event_id):
        return self._call(self.user_id, self.user_type, synced_revision, synced_event_id)


class PurgeContactsUser(ContactsUserOperation):
    db_func_name = 'purge_contacts_user'

    def __call__(self, force=False):
        return self._call(self.user_id, self.user_type, force)


@handle_user_is_not_here
class GetOrCreateTab(UserOperation):
    db_func_name = 'get_or_create_tab'
    result_type = Tab
    change_type = ChangeType.tab_create

    def __call__(self, tab):
        return self._call(
            self.uid, tab, self.request_info
        )


class ResetTabUnvisited(UserOperation):
    db_func_name = 'reset_tab_unvisited'
    change_type = ChangeType.tab_reset_unvisited

    def __call__(self, tab):
        return self._call(self.uid, tab, self.request_info)


class ResetTabFresh(UserOperation):
    db_func_name = 'reset_tab_fresh'
    change_type = ChangeType.tab_reset_unvisited

    def __call__(self, tab):
        return self._call(self.uid, tab, self.request_info)


class DuplicatedCollector(OperationError):
    pass


class CollectorFromMyself(OperationError):
    pass


class CreateCollector(UserOperation):
    db_func_name = 'create_collector'
    result_type = CollectorIdResult
    change_type = ChangeType.collector_create

    internal_errors = (
        (ErrorTextMatcher('Can\'t create collector - already exists'), DuplicatedCollector),
        (ErrorTextMatcher('Can\'t create collector from myself'), CollectorFromMyself)
    )

    def __call__(self, src_uid, auth_token, root_folder_id=None, label_id=None):
        return self._call(self.uid, src_uid, auth_token, root_folder_id, label_id)


class DeleteCollector(UserOperation):
    db_func_name = 'delete_collector'
    change_type = ChangeType.collector_delete

    def __call__(self, collector_id):
        return self._call(self.uid, collector_id)


class UpdateCollectorsMetadata(UserOperation):
    db_func_name = 'update_collectors_metadata'
    change_type = ChangeType.collector_update

    def __call__(self, collector_id, data, need_log):
        return self._call(self.uid, collector_id, Json(data), need_log)

    def _make_query(self, args):
        return "SELECT txid_current(), * FROM %s.%s(%s)" % (
            self.db_schema, self.db_func_name, "%s, %s, %s::jsonb, %s")


class ClearTaskBulkModifySettings(UserOperation):
    q = u"""INSERT INTO settings.bulk_modification
                (id, name, create_date, settings)
            VALUES
                (1, '', NOW(), '{}')
            ON CONFLICT (id) DO UPDATE
            SET name = ''"""

    def __call__(self):
        self._execute_query(self.q, [])
        return self

    db_func_name = None
    result_type = None


class BulkModifySettings(UserOperation):
    db_func_name = 'bulk_modify_settings'

    def __call__(self):
        return self._call()


class CreateTaskBulkModifySettings(UserOperation):
    db_func_name = 'create_task_bulk_modify'

    def __call__(self, name, task_type, settings):
        return self._call(name, task_type, settings)


class EraseSettings(UserOperation):
    db_func_name = 'erase_settings'

    def __call__(self, settings):
        return self._call(self.uid, settings)


class AddUserForBulkUpdateSettings(UserOperation):
    q = u"""INSERT INTO settings.users_for_update
                (uid)
            VALUES
                (%s)"""

    def __call__(self):
        self._execute_query(self.q, [self.uid])
        return self

    db_func_name = None
    result_type = None


class NotExistedUser(OperationError):
    pass


class InvalidUserStateTransition(OperationError):
    pass


class UpdateUserState(UserOperation):
    db_func_name = 'update_user_state'

    internal_errors = [
        ('there is no user with uid', NotExistedUser),
        ('no serials selected', NotExistedUser),
        ('transition from', InvalidUserStateTransition),
    ]

    def __call__(self, new_state, notifies_count=None):
        return self._call(self.uid, new_state, notifies_count)


class ExchangeArchiveState(UserOperation):
    db_func_name = 'exchange_archive_state'

    internal_errors = []

    def __call__(self, user_state, from_state, to_state, notice=None):
        return self._call(self.uid, user_state, from_state, to_state, notice)


class PurgeOldBackup(UserOperation):
    db_func_name = 'purge_old_backup'

    def __call__(self, backup_id):
        return self._call(self.uid, backup_id)


class ReserveBackupId(UserOperation):
    db_func_name = 'reserve_backup_id'

    def __call__(self):
        return self._call(self.uid, self.request_info)


class CreateBackup(UserOperation):
    db_func_name = 'create_backup'

    def __call__(self, backup_id, max_messages, use_tabs):
        return self._call(self.uid, backup_id, max_messages, use_tabs, self.request_info)


class CreateRestore(UserOperation):
    db_func_name = 'create_restore'

    def __call__(self, backup_id, created, method, fids_mapping):
        return self._call(self.uid, backup_id, created, method, fids_mapping, self.request_info)


class FillBackup(UserOperation):
    db_func_name = 'fill_backup'

    def __call__(self, backup_id, use_tabs):
        return self._call(self.uid, backup_id, use_tabs, self.request_info)


class RestoreUpdateMids(UserOperation):
    db_func_name = 'restore_update_mids'

    def __call__(self, backup_id, mids_mapping):
        return self._call(self.uid, backup_id, mids_mapping, self.request_info)


class UpdateBackupSettings(UserOperation):
    db_func_name = 'update_backup_settings'
    result_type = BackupSettings

    def __call__(self, settings):
        return self._call(self.uid, settings, self.request_info)


class CompleteRestore(UserOperation):
    db_func_name = 'complete_restore'

    def __call__(self):
        return self._call(self.uid, self.request_info)


class FailBackup(UserOperation):
    db_func_name = 'fail_backup'

    def __call__(self, notice):
        return self._call(self.uid, notice, self.request_info)


class FailRestore(UserOperation):
    db_func_name = 'fail_restore'

    def __call__(self, notice):
        return self._call(self.uid, notice, self.request_info)


class DeactivateBackup(UserOperation):
    db_func_name = 'deactivate_backup'

    def __call__(self):
        return self._call(self.uid, self.request_info)


@handle_user_is_not_here
class StoreDeletedMessage(UserOperation):
    db_func_name = 'store_deleted_message'
    result_type = StoreDeletedResult
    integrity_errors = (
        ('pk_mailbox', MailAlreadyExsits),
        ('messages_st_id_without_mulca_2', StidWithMulca2Prefix),
        ('check_plain_not_empty_mime', EmptyMime),
    )
    change_type = ChangeType.store

    def __call__(
            self, coords,
            headers, recipients,
            attaches, mime):
        return self._call(
            self.uid, coords, headers, recipients,
            attaches, mime, self.request_info
        )


class StickerAlreadyExists(OperationError):
    pass


class MessageForStickerDoesNotExist(OperationError):
    pass


class CreateReplyLaterSticker(UserOperation):
    db_func_name = 'create_reply_later_sticker'
    result_type = CreateReplyLaterStickerResult
    integrity_errors = (
        ('pk_stickers_reply_later', StickerAlreadyExists),
        ('fk_stickers_reply_later_box', MessageForStickerDoesNotExist)
    )
    change_type = ChangeType.sticker_create

    def __call__(self, mid, fid, date, tab):
        return self._call(self.uid, mid, fid, date, tab, self.request_info)


class RemoveReplyLaterSticker(UserOperation):
    db_func_name = 'remove_reply_later_sticker'

    def __call__(self, mid):
        return self._call(self.uid, mid, self.request_info)


class RemoveIncorrectStickers(UserOperation):
    db_func_name = 'remove_incorrect_stickers'

    def __call__(self):
        return self._call(self.uid, self.request_info)
