# coding: utf-8

import six
import logging

from mail.pypg.pypg.common import (
    fetch_as_dicts,
)
from mail.pypg.pypg.query_conf import load_from_package
from mail.pypg.pypg.query_handler import (
    QueryHandler,
    FetchAs,
    FetchHeadExpectOneRowAs,
    LazyFetchAs,
    sync_fetch_as_list)
from pymdb import types

log = logging.getLogger(__name__)


def list_or_none(ItemClass, data):
    if data is not None:
        return [ItemClass(**i) for i in data]
    return None


def mails_fetcher(cur):
    return (
        types.Mail(
            mid=md.pop('mid'),
            coords=types.MailCoordinates(**md),
            headers=types.MailHeaders(**md),
            lids=md.pop('lids'),
            recipients=list_or_none(types.MailRecipient, md.pop('recipients')),
            attaches=list_or_none(types.MailAttach, md.pop('attaches')),
            mime=list_or_none(types.MailMimePart, md.pop('mime')),
            **md
        )
        for md in fetch_as_dicts(cur)
    )


def deleted_mails_fetcher(cur):
    return (
        types.Mail(
            mid=md.pop('mid'),
            coords=types.DeletedCoords(**md),
            headers=types.MailHeaders(**md),
            lids=[],
            recipients=list_or_none(types.MailRecipient, md.pop('recipients')),
            attaches=list_or_none(types.MailAttach, md.pop('attaches')),
            mime=list_or_none(types.MailMimePart, md.pop('mime')),
            pop3=False,
            doom_date=None,
            **md
        )
        for md in fetch_as_dicts(cur)
    )


def filters_fetcher(cur):
    rules = []
    for rd in fetch_as_dicts(cur):
        conditions = rd.pop('conditions') or []
        conditions = [types.FilterCondition(**c) for c in conditions]
        actions = rd.pop('actions') or []
        actions = [types.FilterAction(**a) for a in actions]
        rules.append(
            types.Filter(
                actions=actions,
                conditions=conditions,
                **rd
            )
        )
    return rules


def simple_filters_fetcher(cur):
    rules = []
    for rd in fetch_as_dicts(cur):
        rules.append(
            types.Filter(
                **rd
            )
        )
    return rules


def get_head(seq):
    return iter(seq).next()


def sync_fetch_as_dicts(cur):
    return list(fetch_as_dicts(cur))


def fetch_one_row_with_one_element(cur):
    return cur.fetchone()[0]


def dict_handler(query):
    return (query, sync_fetch_as_dicts)


class FolderDoesNotExist(AssertionError):
    pass


class ExpectOneFolderError(AssertionError):
    pass


class FetchFirstCell(object):
    def __call__(self, cur):
        result = cur.fetchone()
        if not result:
            return None
        return result[0]


class TabDoesNotExist(AssertionError):
    pass


class ExpectOneTabError(AssertionError):
    pass


class BackupDoesNotExist(AssertionError):
    pass


class ExpectOneBackupError(AssertionError):
    pass


class Queries(QueryHandler):
    q = load_from_package(__package__, __file__)

    handlers = dict(
        user_exists=(q.user_exists, FetchFirstCell()),
        user=(q.user, FetchHeadExpectOneRowAs(types.User)),
        shared_fids=(q.shared_fids, sync_fetch_as_list),
        shared_folders=(q.shared_folders, FetchAs(types.SharedFolder)),
        shared_folder_subscriptions=(
            q.shared_folder_subscriptions,
            FetchAs(types.SharedFolderSubscription)),
        shared_folder_change_queue=(
            q.shared_folder_change_queue,
            sync_fetch_as_list),
        subscribed_folders=(
            q.subscribed_folders,
            FetchAs(types.SubscribedFolder)),
        _synced_messages=(
            q.synced_messages,
            LazyFetchAs(types.SyncedMessage)),
        folders=(q.folders, FetchAs(types.Folder)),
        pop3_folders=(q.pop3_folders, FetchAs(types.Folder)),
        labels=(q.labels, FetchAs(types.Label)),
        default_folders=(q.default_folders, FetchAs(types.DefaultFolder)),
        default_labels=(q.default_labels, FetchAs(types.MailLabelDef)),
        doomed_folder_types=(q.doomed_folder_types, sync_fetch_as_list),
        threaded_folder_types=(q.threaded_folder_types, sync_fetch_as_list),
        get_purge_steps=(
            q.get_purge_steps,
            fetch_one_row_with_one_element),
        get_purge_on_delete_steps=(
            q.get_purge_on_delete_steps,
            fetch_one_row_with_one_element),
        imap_unsubscribed_folders=(
            q.imap_unsubscribed_folders,
            FetchAs(types.IMAPUnsubscribedFolder)),
        _mails=(q.mails_dicts, mails_fetcher),
        _message_references=(
            q.message_references,
            LazyFetchAs(types.MessageReference)),
        _threads_hashes=(q.threads_hashes, LazyFetchAs(types.ThreadsHash)),
        filter_elists=(q.filter_elists, FetchAs(types.FilterElist)),
        filters=(q.rules_dicts, filters_fetcher),
        fix_log=(q.fix_log, FetchAs(types.FixLog)),
        counters=(q.counters, FetchHeadExpectOneRowAs(types.Counters)),
        serials=(q.serials, FetchHeadExpectOneRowAs(types.Serials)),
        default_serials=(q.default_serials, FetchHeadExpectOneRowAs(types.Serials)),
        _deleted_mails=(q.deleted_mails, deleted_mails_fetcher),
        _deleted_mails_dict=dict_handler(q.deleted_mails),
        deleted_messages=(q.deleted_messages, FetchAs(types.DeletedMessages)),
        storage_delete_queue=dict_handler(q.storage_delete_queue),
        messages=dict_handler(q.messages),
        message=(q.message, FetchHeadExpectOneRowAs(dict)),
        references=(q.references, sync_fetch_as_list),
        threads=dict_handler(q.threads),
        imap_messages=dict_handler(q.imap_messages),
        changelog=dict_handler(q.changelog),
        chained_log=dict_handler(q.chained_log),
        violations=dict_handler(q.violations),
        folder_path=(q.folder_path, fetch_one_row_with_one_element),
        pop3_box=dict_handler(q.pop3_box),
        mids_with_lids=(q.mids_with_lids, sync_fetch_as_list),
        mids_without_lids=(q.mids_without_lids, sync_fetch_as_list),
        messages_table=dict_handler(q.messages_table),
        chained_rec_count=(q.chained_rec_count, FetchHeadExpectOneRowAs(dict)),
        mailish_accounts=(q.mailish_accounts, FetchAs(types.MailishAccount)),
        mailish_auth_data=(q.mailish_auth_data, FetchAs(types.MailishAuthData)),
        mailish_folders=(q.mailish_folders, FetchAs(types.MailishFolder)),
        _mailish_messages=(q.mailish_messages, LazyFetchAs(types.MailishMessage)),
        mailish_data=(q.mailish_data, sync_fetch_as_dicts),
        mailish_count=(q.mailish_count, sync_fetch_as_dicts),
        mailish_folders_count=(q.mailish_folders_count, sync_fetch_as_dicts),
        mailish_auth_count=(q.mailish_auth_count, sync_fetch_as_dicts),
        mailish_security_locks_count=(q.mailish_security_locks_count, sync_fetch_as_dicts),
        mailish_folder=(q.mailish_folder, sync_fetch_as_dicts),
        mailish_data_by_imap_coords=(q.mailish_data_by_imap_coords, sync_fetch_as_dicts),
        mailish_auth_data_by_token_id=(q.mailish_auth_data_by_token_id, sync_fetch_as_dicts),
        rules_get=(q.rules_get, simple_filters_fetcher),
        _windat_messages=(q.windat_messages, LazyFetchAs(types.WinDatMessages)),
        unsubscribe_tasks=(
            q.unsubscribe_tasks,
            FetchAs(types.UnsubscribeTask)),
        folder_archivation_rules=(
            q.folder_archivation_rules,
            FetchAs(types.FolderArchivationRule)),
        settings=(q.get_settings, FetchAs(types.Settings)),
        contacts_user=(q.get_contacts_user, FetchHeadExpectOneRowAs(types.ContactsUserExt)),
        contacts_serials=(q.get_contacts_serials, FetchHeadExpectOneRowAs(types.ContactsSerials)),
        contacts_list_by_type_and_name=(
            q.get_contacts_list_by_type_and_name,
            FetchAs(types.ContactsList),
        ),
        contacts_list_by_id=(
            q.get_contacts_list_by_id,
            FetchAs(types.ContactsList),
        ),
        contacts_tag_by_type_and_name=(
            q.get_contacts_tag_by_type_and_name,
            FetchAs(types.ContactsTag),
        ),
        contacts_tag_by_id=(
            q.get_contacts_tag_by_id,
            FetchAs(types.ContactsTag),
        ),
        contact_by_id=(
            q.get_contact_by_id,
            FetchAs(types.Contact),
        ),
        contact_tag=(
            q.get_contact_tag,
            FetchAs(types.ContactTag),
        ),
        shared_contacts_list_to=(
            q.get_shared_contacts_list_to,
            FetchAs(types.SharedContactsListTo),
        ),
        subscribed_contacts_list_to=(
            q.get_subscribed_contacts_list_to,
            FetchAs(types.SubscribedContactsListTo),
        ),
        contacts_change_log=dict_handler(q.get_contacts_change_log),
        contacts_email_by_id=(
            q.get_contacts_email_by_id,
            FetchAs(types.ContactsEmail),
        ),
        contacts_email_tag=(
            q.get_contacts_email_tag,
            FetchAs(types.ContactsEmailTag),
        ),
        passport_user_contacts_user=(
            q.get_passport_user_contacts_users,
            FetchHeadExpectOneRowAs(types.ContactsUser)
        ),
        passport_user_contacts_users=(
            q.get_passport_user_contacts_users,
            FetchAs(types.ContactsUser),
        ),
        passport_user_contacts_serials=(
            q.get_passport_user_contacts_serials,
            FetchAs(types.ContactsSerials),
        ),
        passport_user_contacts_lists=(
            q.get_passport_user_contacts_lists,
            FetchAs(types.ContactsList),
        ),
        passport_user_contacts_tags=(
            q.get_passport_user_contacts_tags,
            FetchAs(types.ContactsTag),
        ),
        passport_user_contacts_contacts=(
            q.get_passport_user_contacts_contacts,
            FetchAs(types.Contact),
        ),
        passport_user_contacts_emails=(
            q.get_passport_user_contacts_emails,
            FetchAs(types.ContactsEmail),
        ),
        passport_user_contacts_contacts_tags=(
            q.get_passport_user_contacts_contacts_tags,
            FetchAs(types.ContactTag),
        ),
        passport_user_contacts_emails_tags=(
            q.get_passport_user_contacts_emails_tags,
            FetchAs(types.ContactsEmailTag),
        ),
        passport_user_contacts_shared_lists=(
            q.get_passport_user_contacts_shared_lists,
            FetchAs(types.SharedContactsListTo),
        ),
        passport_user_default_contacts_list_id=(
            q.get_passport_user_default_contacts_list_id,
            sync_fetch_as_list,
        ),
        passport_user_contacts_subscribed_lists=(
            q.get_passport_user_contacts_subscribed_lists,
            FetchAs(types.SubscribedContactsListTo),
        ),
        tabs=(q.tabs, FetchAs(types.Tab)),
        messages_by_tab=dict_handler(q.messages_by_tab),
        threads_by_tab=dict_handler(q.threads_by_tab),
        collectors=(q.collectors, FetchAs(types.Collector)),
        collectors_metadata=dict_handler(q.collectors_metadata),
        attach_counters=(q.attach_counters, FetchHeadExpectOneRowAs(types.AttachCounters)),
        stats_users_info=(q.stats_users_info, FetchAs(types.UserStats)),
        bulk_task_modify_settings=(q.get_bulk_task_modify_settings, FetchAs(types.BulkTaskModifySettings)),
        user_for_update_settings=(q.get_user_for_update_settings, FetchAs(types.UserForModifySettings)),
        user_for_init_settings=(q.get_user_for_init_settings, FetchAs(types.UserForModifySettings)),
        message_count=(q.message_count, sync_fetch_as_dicts),
        backups=(q.backups, FetchAs(types.Backup)),
        restores=(q.restores, FetchAs(types.Restore)),
        backup_folders=(q.backup_folders, FetchAs(types.BackupFolders)),
        backup_box=(q.backup_box, FetchAs(types.BackupBox)),
        folders_to_backup=(q.folders_to_backup, FetchAs(types.FoldersToBackup)),
        tabs_to_backup=(q.tabs_to_backup, FetchAs(types.TabsToBackup)),
        archives=(q.archives, FetchAs(types.Archives)),
        stickers_reply_later=(q.stickers_reply_later, FetchAs(types.StickersReplyLater)),
    )

    def __init__(self, conn, uid):
        super(Queries, self).__init__(conn)
        self.uid = uid

    def __repr__(self):
        return 'Queries(uid:{0}, conn:{1})'.format(self.uid, self.conn)

    def _join_qargs(self, query, qargs):
        qargs = qargs or {}
        if 'uid' in query.args and 'uid' not in qargs:
            qargs['uid'] = self.uid
        return super(Queries, self)._join_qargs(query, qargs)

    def _folder_by_attribute(self, attr, val):
        all_folders = self.folders()
        fld = [f for f in all_folders if getattr(f, attr) == val]
        if not fld:
            raise FolderDoesNotExist(
                "Can't find folder with {attr} {val}"
                " in user folders: {all_folders}".format(
                    **locals()))
        if len(fld) > 1:
            raise ExpectOneFolderError(
                "Found {count} folders {fld} with {attr} {val}"
                " in user folders: {all_folders}".format(
                    count=len(fld), **locals()))
        return fld[0]

    def reply_later_sticker_on_mid(self, mid):
        return len([s for s in self.stickers_reply_later() if s.mid == mid]) > 0

    def size_of_reply_later_stickers(self):
        return len(self.stickers_reply_later())

    def folder_by_attribute(self, attr, val):
        if attr == 'name':
            return self.folder_by_name(val)
        return self._folder_by_attribute(attr, val)

    def folder_by_id(self, fid):
        return self._folder_by_attribute('fid', fid)

    def folder_by_name(self, folder_name):
        all_folders = self.folders()
        parents_fids = [None]
        fld = []
        for short_name in folder_name.split('|'):
            fld = [
                f for f in all_folders
                if f.name == short_name and f.parent_fid in parents_fids
            ]
            parents_fids = [f.fid for f in fld]

        if not fld:
            raise FolderDoesNotExist(
                "Can't find folder with name {folder_name}"
                " in user folders: {all_folders}, tree: {parents_fids}".format(
                    **locals()))
        if len(fld) > 1:
            raise ExpectOneFolderError(
                "Found {count} folders {fld} with name {folder_name}"
                " in user folders: {all_folders}".format(
                    count=len(fld), **locals()))
        return fld[0]

    def folder_by_type(self, folder_type):
        return self._folder_by_attribute('type', folder_type)

    def folder_by(self, folder_type, folder_name):
        if folder_type is not None:
            return self.folder_by_type(folder_type)
        elif folder_name is not None:
            return self.folder_by_name(folder_name)
        else:
            AssertionError('No folder identity to select')

    def revision(self, revision):
        if six.PY2:
            if isinstance(revision, (int, long)):  # noqa
                return revision
            assert isinstance(revision, (str, unicode))  # noqa
        else:
            if isinstance(revision, int):
                return revision
        assert isinstance(revision, str)
        if revision.isdigit():
            return int(revision)
        assert revision == "HEAD",\
            "unknown revision format: {0}".format(revision)
        return self.global_revision()

    def find_labels(self, labels_def):
        all_labels = self.labels()
        requested = [l for l in all_labels if l in labels_def]
        assert len(requested) == len(labels_def), \
            "Can't find some labels requested: {0}, found: {1}".format(
                labels_def, requested)
        return requested

    def find_one_label(self, label_def):
        return self.find_labels([label_def])[0]

    def fresh_counter(self):
        return self.counters().fresh_count

    def counters_revision(self):
        return self.counters().revision

    def global_revision(self):
        return self.serials().next_revision - 1

    def blacklist(self):
        return [
            f.as_dict()
            for f in self.filter_elists()
            if f.list == 'black']

    def mails(self, min_received_date=None, max_received_date=None, **kwargs):
        return self.eager('_mails')(
            min_received_date=min_received_date,
            max_received_date=max_received_date,
            **kwargs
        )

    def lazy_mails(self, min_received_date=None, max_received_date=None, **kwargs):
        return self.lazy('_mails')(
            min_received_date=min_received_date,
            max_received_date=max_received_date,
            **kwargs
        )

    def deleted_mails(self, min_received_date=None, max_received_date=None, **kwargs):
        return self.lazy('_deleted_mails')(
            min_received_date=min_received_date,
            max_received_date=max_received_date,
            **kwargs
        )

    def deleted_mails_dict(self, min_received_date=None, max_received_date=None, **kwargs):
        return self._deleted_mails_dict(
            min_received_date=min_received_date,
            max_received_date=max_received_date,
            **kwargs
        )

    def synced_messages(self, min_received_date=None, max_received_date=None, **kwargs):
        return self.eager('_synced_messages')(
            min_received_date=min_received_date,
            max_received_date=max_received_date,
            **kwargs
        )

    def lazy_synced_messages(self, min_received_date=None, max_received_date=None, **kwargs):
        return self.lazy('_synced_messages')(
            min_received_date=min_received_date,
            max_received_date=max_received_date,
            **kwargs
        )

    def message_references(self, min_received_date=None, max_received_date=None, **kwargs):
        return self.lazy('_message_references')(
            min_received_date=min_received_date,
            max_received_date=max_received_date,
            **kwargs
        )

    def threads_hashes(self, min_received_date=None, max_received_date=None, **kwargs):
        return self.lazy('_threads_hashes')(
            min_received_date=min_received_date,
            max_received_date=max_received_date,
            **kwargs
        )

    def windat_messages(self, min_received_date=None, max_received_date=None, **kwargs):
        return self.lazy('_windat_messages')(
            min_received_date=min_received_date,
            max_received_date=max_received_date,
            **kwargs
        )

    def mailish_messages(self, min_received_date=None, max_received_date=None, **kwargs):
        return self.lazy('_mailish_messages')(
            min_received_date=min_received_date,
            max_received_date=max_received_date,
            **kwargs
        )

    def tab_by_type(self, tab_type):
        all_tabs = self.tabs()
        tab = [t for t in all_tabs if t.tab == tab_type]
        if not tab:
            raise TabDoesNotExist(
                "Can't find tab with type {tab_type}"
                " in user tabs: {all_tabs}".format(
                    **locals()))
        if len(tab) > 1:
            raise ExpectOneTabError(
                "Found {count} tabs {tab} with type {tab_type}"
                " in user folders: {all_tabs}".format(
                    count=len(tab), **locals()))
        return tab[0]

    def backup_by_id(self, backup_id):
        all_backups = self.backups()
        backup = [b for b in all_backups if b.backup_id == backup_id]
        if not backup:
            raise BackupDoesNotExist(
                "Can't find backup with id {backup_id}"
                " in user backups: {all_backups}".format(
                    **locals()))
        if len(backup) > 1:
            raise ExpectOneBackupError(
                "Found {count} backups {backup} with id {backup_id}"
                " in user backups: {all_backups}".format(
                    count=len(backup), **locals()))
        return backup[0]

    def backups_by_state(self, state):
        return [b for b in self.backups() if b.state == state]

    def __filter_by_backup_id(self, query, backup_id):
        all_rows = getattr(self, query)()
        return [r for r in all_rows if r.backup_id == backup_id]

    def restores_by_id(self, backup_id):
        return self.__filter_by_backup_id('restores', backup_id)

    def backup_folders_by_id(self, backup_id):
        return self.__filter_by_backup_id('backup_folders', backup_id)

    def backup_box_by_id(self, backup_id):
        return self.__filter_by_backup_id('backup_box', backup_id)

    def backup_settings(self):
        folders = self.folders_to_backup()
        tabs = self.tabs_to_backup()
        return types.BackupSettings(
            fids=[f.fid for f in folders],
            tabs=[t.tab for t in tabs],
        )
