# coding: utf-8
"""
Simple parser for test_decoding message

https://www.postgresql.org/docs/9.6/static/test-decoding.html
"""

from collections import namedtuple
import tatsu


def _load_gramma():
    from library.python import resource
    gramma_ebnf = resource.find(
        'resfs/file/mail/pg/mdb/python/pymdb/replication/postgre_test_decoding.ebnf'
    ).decode('utf-8')
    return tatsu.compile(gramma_ebnf)


_gramma = _load_gramma()

Begin = namedtuple('Begin', ['xid'])
Commit = namedtuple('Commit', ['xid'])
Attribute = namedtuple('Attribute', ['type', 'value'])
DML = namedtuple('DML', ['table', 'command', 'attributes'])


class Parser(object):
    class Semantics(object):
        @staticmethod
        def xid(ast):
            return int(ast)

        @staticmethod
        def begin(ast):
            return Begin(ast[1])

        @staticmethod
        def commit(ast):
            return Commit(ast[1])

        @staticmethod
        def no_tuple_data(ast):
            return {}

        @staticmethod
        def int_value(ast):
            return int(ast)

        @staticmethod
        def boolean_value(ast):
            return ast in (u'true', u'yes')

        @staticmethod
        def null_value(ast):
            return None

        @staticmethod
        def quoted_value(ast):
            return ast.strip(u"'")

        @staticmethod
        def dml(ast):
            table, command, attributes = ast
            return DML(
                table=table,
                command=command,
                attributes=dict((a[0], Attribute(a[1], a[2])) for a in attributes)
            )

    def __init__(self):
        self._gramma = _load_gramma()

    def parse_message(self, data):
        return self._gramma.parse(
            data,
            semantics=self.Semantics)
