# coding: utf-8
from collections import namedtuple
import operator
from datetime import datetime

from mail.pypg.pypg.types.selected import BaseSelectedType as PPBaseSelectedType

from .mixin import LabelCompareMixin, CmpableMixin
from .adapted import Pop3State


class BaseSelectedType(PPBaseSelectedType, CmpableMixin):
    _volatile = [
        'created',
        'mid',
        'tid',
        'lid',
        'fid'
    ]

    @classmethod
    def cmpable_keys(cls):
        return [k for k in cls.__slots__ if k not in cls._volatile]


class User(BaseSelectedType):
    __slots__ = (
        'here_since', 'is_here', 'data_version', 'is_deleted', 'purge_date', 'can_read_tabs',
        'state', 'last_state_update', 'notifies_count',
    )
    _volatile = ['here_since']


class Mail(BaseSelectedType):    # pylint: disable=R0903
    __slots__ = (
        'mid',
        'coords',
        'headers',
        'lids',
        'recipients',
        'attaches',
        'mime',
        'pop_uidl',
        'pop3',
        'doom_date',
    )

    _volatile = ['mid', 'lids']


class Folder(BaseSelectedType):
    __slots__ = (
        'fid', 'type', 'name', 'parent_fid',
        'revision',
        'message_count', 'message_seen', 'message_recent',
        'message_size',
        'first_unseen', 'first_unseen_id',
        'attach_count', 'attach_size',
        'unvisited', 'unique_type',
        'uidvalidity', 'created',
        'next_imap_id', 'pop3state', 'position',
        'subscribed_for_shared_folder'
    )

    _defaults = dict(
        parent_fid=None,
        next_imap_id=1,
        unvisited=False,
        message_count=0,
        message_seen=0,
        message_recent=0,
        message_size=0,
        attach_count=0,
        attach_size=0,
        first_unseen=0,
        first_unseen_id=None,
        pop3state=Pop3State(initialized=False, enabled=False),
        position=0,
        subscribed_for_shared_folder=None,
    )

    _volatile = [
        'fid',
        'parent_fid',
        'uidvalidity',
        'created',
        'first_unseen',
        'first_unseen_id',
    ]

    USER = 'user'
    INBOX = 'inbox'
    SPAM = 'spam'
    TRASH = 'trash'
    SENT = 'sent'
    OUTBOX = 'outbox'
    PENDING = 'pending'
    DRAFTS = 'drafts'
    ARCHIVE = 'archive',
    TEMPLATES = 'templates'
    DISCOUNT = 'discount'

    @property
    def message_unseen(self):
        return self.message_count - self.message_seen


class Label(LabelCompareMixin, BaseSelectedType):
    __slots__ = (
        'lid', 'name', 'type', 'message_count', 'message_seen',
        'color', 'created', 'revision'
    )

    _defaults = {
        'revision': 1,
        'message_count': 0,
        'message_seen': 0,
        'color': None,
        'created': datetime.now
    }

    DOMAIN = 'domain'
    USER = 'user'
    SYSTEM = 'system'
    TYPE = 'type'
    ATTRIBUTE = 'attribute'


class IMAPUnsubscribedFolder(BaseSelectedType):
    __slots__ = ('full_name', 'revision')


class ThreadInfo(BaseSelectedType):
    __slots__ = ('tid', 'message_count')


class StoreResult(BaseSelectedType):
    __slots__ = ('mid', 'tid', 'imap_id', 'revision')


class SyncResult(BaseSelectedType):
    __slots__ = ('mid', 'revision', 'owner_mid', 'owner_revision')


class StoreDeletedResult(BaseSelectedType):
    __slots__ = ('mid', 'revision')


class RegisterUserResult(BaseSelectedType):
    __slots__ = ['register_user']


class RestoreResult(BaseSelectedType):
    __slots__ = ('deleted_mid', 'why_not_restored', 'restored_mid')

    @property
    def mid(self):
        return self.restored_mid


class QuickSaveResult(BaseSelectedType):
    __slots__ = ('updated', 'revision')


class OperationResult(BaseSelectedType):
    __slots__ = (
        'revision', 'mids',
    )

    @property
    def mid(self):
        assert len(self.mids) == 1,\
            'Ask for mids, but {0} contains more then one mid'.format(self)
        return self.mids[0]


class CopyResult(BaseSelectedType):
    __slots__ = ('mid', 'revision')


class CreateFolderResult(BaseSelectedType):
    __slots__ = ('uid', 'fid', 'revision', 'name', 'parent_fid', 'type',
                 'unique_type', 'created',
                 'next_imap_id', 'uidvalidity', 'first_unseen',
                 'first_unseen_id', 'message_count', 'message_seen',
                 'message_size', 'message_recent', 'attach_count',
                 'attach_size', 'unvisited', 'position')


class FilterCondition(BaseSelectedType):
    __slots__ = ('field_type', 'field', 'pattern',
                 'oper', 'link', 'negative')

    def __eq__(self, other):
        return all(
            operator.eq(
                getattr(self, a),
                getattr(other, a))
            for a in self.__slots__
        )


class FilterAction(BaseSelectedType):
    __slots__ = ('action_id', 'oper', 'param', 'verified')
    _defaults = dict(
        action_id=None
    )
    _volatile = ['action_id', 'param']

    def __eq__(self, other):
        return all(
            operator.eq(
                getattr(self, a),
                getattr(other, a))
            for a in self.__slots__[1:]
        )


class Filter(BaseSelectedType):
    __slots__ = ('rule_id', 'name', 'enabled',
                 'prio', 'stop', 'created', 'type',
                 'conditions', 'actions',)
    _defaults = dict(
        rule_id=None,
        type='user',
    )
    _volatile = BaseSelectedType._volatile + ['rule_id']

    def __init__(self, **kwargs):
        if 'actions' not in kwargs:
            kwargs['actions'] = []
        if 'conditions' not in kwargs:
            kwargs['conditions'] = []
        super(Filter, self).__init__(**kwargs)


class FilterElist(BaseSelectedType):
    __slots__ = (
        'email',
        'list',
        'created'
    )


class Counters(BaseSelectedType):
    __slots__ = (
        'fresh_count',
        'revision',
    )
    _defaults = {
        'revision': 1
    }


class MessageReference(BaseSelectedType):
    __slots__ = (
        'mid',
        'type',
        'value',
    )

    REFERENCE_TYPE = 'reference'
    IN_REPLY_TO_TYPE = 'in-reply-to'


class ThreadsHash(BaseSelectedType):
    __slots__ = (
        'tid',
        'namespace',
        'value',
        'uniq_key'
    )
    _volatile = [
        'uniq_key',
        'tid',
    ]

    SUBJ_NAMESPACE = 'subject'
    FROM_NAMESPACE = 'from'

MailMessagesCoords = (
    'st_id',
    'size',
    'attributes',
    'pop_uidl',
    'found_tid',
    'thread_rule',
)
"""Fields of mail.messages table that are being used in MailCoords classes"""


class MailCoordinates(BaseSelectedType):
    """
    mail.box entry reflection
    """
    __slots__ = (
        'fid',
        'tab',
        'tid',
        'imap_id',
        # 'newest_tif',
        'revision',
        # 'chain',
        'seen',
        'recent',
        'deleted',
        'received_date',
    ) + MailMessagesCoords
    _volatile = ['fid', 'tid', 'imap_id']


class MailHeaders(BaseSelectedType):
    __slots__ = (
        'subject',
        'firstline',
        'hdr_date',
        'hdr_message_id',
        'extra_data')


class Serials(BaseSelectedType):
    __slots__ = (
        'next_revision',
        'next_mid_serial',
        'next_lid', 'next_fid',
        'next_owner_subscription_id',
        'next_subscriber_subscription_id',
        'next_collector_id',
        'next_backup_id'
    )


class FixLog(BaseSelectedType):
    __slots__ = (
        'revision',
        'fix_key',
        'fixed',
        'fix_date',
        'db_user',
    )


class DeletedMessages(BaseSelectedType):
    __slots__ = (
        'mid',
        'revision',
        'deleted_date',
        'info',
        'received_date',
    )


class DeletedCoords(BaseSelectedType):
    """
    mail.deleted_box entry reflection
    """
    __slots__ = (
        'revision',
        'deleted_date',
        'info',
        'received_date',
    ) + MailMessagesCoords

    @classmethod
    def cmpable_keys(cls):
        return [k for k in cls.__slots__ if (k not in cls._volatile and k != 'info')]


ThreadLabel = namedtuple('ThreadLabel', ('lid', 'message_count'))

DefaultFolder = namedtuple('DefaultFolder', ('type', 'name', 'imap_unsubscribed'))


class SharedFolder(BaseSelectedType):
    """
    mail.shared_folder entry
    """
    __slots__ = ('fid', 'revision', 'created')

    _defaults = {
        'revision': 1,
        'created': datetime.now
    }


class SharedFolderSubscription(BaseSelectedType):
    """
    mail.shared_folder_subscriptions entry
    """
    __slots__ = (
        'subscription_id', 'fid', 'subscriber_uid',
        'created', 'updated',
        'worker_id', 'state', 'fail_reason',
    )


class SubscribedFolder(BaseSelectedType):
    """
    mail.subscribed_folders entry

    user subscription
    """
    __slots__ = (
        'fid', 'revision',
        'owner_uid', 'owner_fid',
        'subscription_id',
        'synced_revision',
        'created',
        'synced_imap_id',
    )


class SyncedMessage(BaseSelectedType):
    """
    mail.synced_messages entry
    """
    __slots__ = (
        'mid', 'tid', 'revision',
        'subscription_id', 'owner_mid', 'owner_tid', 'owner_revision',
    )


class MailishAccount(BaseSelectedType):
    __slots__ = (
        'email',
        'imap_login',
        'imap_server',
        'imap_port',
        'imap_ssl',
        'smtp_login',
        'smtp_server',
        'smtp_port',
        'smtp_ssl',
        'last_sync',
    )


class MailishAuthData(BaseSelectedType):
    __slots__ = (
        'token_id',
        'auth_type',
        'uuid',
        'lock_flag',
        'last_valid',
        'oauth_app',
        'imap_credentials',
        'smtp_credentials',
    )


class MailishFolder(BaseSelectedType):
    __slots__ = (
        'fid',
        'imap_path',
        'uidvalidity',
        'range_start',
        'range_end',
    )

    _volatile = ['fid']


class MailishMessage(BaseSelectedType):
    __slots__ = (
        'fid',
        'imap_id',
        'imap_time',
        'mid',
        'errors',
    )

    _volatile = ['fid', 'mid']


class WinDatMessages(BaseSelectedType):
    __slots__ = (
        'mid',
        'st_id',
        'hid',
        'windat',
    )


class UnsubscribeTask(BaseSelectedType):
    __slots__ = (
        'task_id',
        'task_request_id',
        'owner_uid',
        'owner_fids',
        'subscriber_uid',
        'root_subscriber_fid',
        'assigned',
    )


class FolderArchivationRule(BaseSelectedType):
    """
    mail.folder_archivation_rules entry
    """
    __slots__ = (
        'fid',
        'revision',
        'archive_type',
        'keep_days',
        'created',
        'max_size',
    )


class Settings(BaseSelectedType):
    __slots__ = (
        'value',
    )


class ContactsUser(BaseSelectedType):
    __slots__ = (
        'here_since',
        'is_here',
        'is_deleted',
    )


# TODO: remove after MAILPG-2240
class ContactsUserExt(BaseSelectedType):
    __slots__ = (
        'here_since',
        'is_here',
        'is_deleted',
        'is_directory_sync_enabled',
        'directory_synced_revision',
        'directory_last_event_id',
        'directory_last_synced_event_id',
        'directory_last_sync_date',
        'directory_pending_events_count',
    )


class ContactsSerials(BaseSelectedType):
    __slots__ = (
        'next_list_id',
        'next_contact_id',
        'next_tag_id',
        'next_revision',
        'next_email_id',
    )


class ContactsList(BaseSelectedType):
    __slots__ = (
        'list_id',
        'revision',
        'name',
        'type',
        'unique_type',
    )


class ContactsTag(BaseSelectedType):
    __slots__ = (
        'tag_id',
        'revision',
        'name',
        'type',
    )


class Contact(BaseSelectedType):
    __slots__ = (
        'contact_id',
        'list_id',
        'revision',
        'format',
        'vcard',
        'uri',
    )


class ContactTag(BaseSelectedType):
    __slots__ = (
        'contact_id',
        'tag_id',
        'revision',
    )


class SharedContactsListTo(BaseSelectedType):
    __slots__ = (
        'list_id',
        'client_user_id',
        'client_user_type',
        'revision',
    )


class SubscribedContactsListTo(BaseSelectedType):
    __slots__ = (
        'list_id',
        'owner_user_id',
        'owner_user_type',
        'owner_list_id',
        'revision'
    )


class ContactsEmail(BaseSelectedType):
    __slots__ = (
        'email_id',
        'contact_id',
        'revision',
        'email',
        'type',
        'label'
    )


class ContactsEmailTag(BaseSelectedType):
    __slots__ = (
        'contact_id',
        'email_id',
        'tag_id',
        'revision',
    )


class ImapIdResult(BaseSelectedType):
    __slots__ = ('revision', 'imap_id')


class Tab(BaseSelectedType):
    __slots__ = (
        'tab',
        'revision',
        'unvisited',
        'created',
        'message_count',
        'message_seen',
        'message_size',
        'attach_count',
        'attach_size',
        'fresh_count',
    )

    @property
    def message_unseen(self):
        return self.message_count - self.message_seen


class UserStats(BaseSelectedType):
    __slots__ = (
        'last_update',
        'db_size',
        'storage_size',
    )


class Collector(BaseSelectedType):
    __slots__ = ('collector_id', 'metadata')


class CollectorIdResult(BaseSelectedType):
    __slots__ = ('revision', 'collector_id')


class AttachCounters(BaseSelectedType):
    __slots__ = (
        'has_attaches_count',
        'has_attaches_seen',
    )

    @property
    def has_attaches_unseen(self):
        return self.has_attaches_count - self.has_attaches_seen


class BulkTaskModifySettings(BaseSelectedType):
    __slots__ = (
        'name',
    )


class UserForModifySettings(BaseSelectedType):
    __slots__ = (
        'is_modified',
    )


class Backup(BaseSelectedType):
    __slots__ = (
        'backup_id',
        'version',
        'message_count',
        'state',
        'revision',
        'created',
        'updated',
        'notice',
    )


class Restore(BaseSelectedType):
    __slots__ = (
        'backup_id',
        'created',
        'updated',
        'to_restore_count',
        'restored_count',
        'state',
        'method',
        'notice',
        'fids_mapping',
    )


class BackupFolders(BaseSelectedType):
    __slots__ = (
        'backup_id',
        'fid',
        'type',
        'name',
        'parent_fid',
        'is_backuped',
    )


class BackupBox(BaseSelectedType):
    __slots__ = (
        'backup_id',
        'mid',
        'st_id',
        'fid',
        'tab',
        'received_date',
    )


class FoldersToBackup(BaseSelectedType):
    __slots__ = ('fid',)


class TabsToBackup(BaseSelectedType):
    __slots__ = ('tab',)


class Archives(BaseSelectedType):
    __slots__ = (
        'version',
        'state',
        'revision',
        'message_count',
        'restored_message_count',
        'updated',
        'notice',
    )


class StickersReplyLater(BaseSelectedType):
    __slots__ = (
        'mid',
        'fid',
        'date',
        'created',
        'tab',
    )


CreateReplyLaterStickerResult = StickersReplyLater
