# coding: utf-8
from __future__ import print_function

import logging
from argparse import ArgumentParser

from mail.pypg.pypg.common import transaction

from ora2pg.app.config_file import (
    env_to_config_file,
    load_config_file
)
from ora2pg.app.logs import init_logging
from ora2pg.app.transfer_app import dict2args
from ora2pg.blackbox import (
    get_all_by_login,
    get_all_by_uid,
    NonExistentUserError,
)
from ora2pg.copy_user import pg2pg as copy_user_between_pg
from ora2pg.sharpei import (get_pg_dsn_from_sharpei,
                            init_in_sharpei)
from ora2pg.transfer_data import (
    UserInEndpoint,
    DbEndpoint,
    get_transfer_info,
)
from pymdb import operations as OPS
from pymdb.queries import Queries as MdbQueries
from tests_common import fbbdb

log = logging.getLogger(__name__)


def get_env_config(name):
    return load_config_file(env_to_config_file(name))

LOAD = dict2args(get_env_config('load'))
DOCKER = dict2args(get_env_config('docker'))
MDB_DSN = 'host=maildb dbname=maildb'


def create_user_in_docker(uid, login):
    with transaction('host=fbbdb dbname=fbbdb') as conn:
        try:
            user = get_all_by_uid(DOCKER.blackbox, uid)
        except NonExistentUserError:
            user = fbbdb.user_from_id(uid, login)
            fbbdb.add_user(conn, user)
        init_in_sharpei(
            uid, DOCKER.sharddb, True, 1
        )
        return user


def copy_from_load(args):
    source_user = get_all_by_login(LOAD.blackbox, args.login)

    source_dsn = get_pg_dsn_from_sharpei(
        LOAD.sharpei, source_user.uid, LOAD.maildb_dsn_suffix
    )

    dest_user = create_user_in_docker(args.uid or source_user.uid, source_user.login)

    with transaction(source_dsn) as from_pg_conn:
        copy_user_between_pg(
            from_pg_conn=from_pg_conn,
            to_pg_dsn=MDB_DSN,
            transfer_info=get_transfer_info(
                src=UserInEndpoint(
                    db=DbEndpoint.make_pg(1),
                    uid=source_user.uid,
                    suid=None,
                ),
                dst=UserInEndpoint(
                    db=DbEndpoint.make_pg(1),
                    uid=dest_user.uid,
                    suid=None,
                ),
            ),
            pop3_enabled=False
        )


def init_user_if_not_exists(mdb_conn, uid):
    qs = MdbQueries(mdb_conn, uid)
    if not qs.folders():
        print('Initialize user %r' % uid)
        init_op = OPS.Init(mdb_conn, uid)
        init_op()


def subscriber_user_to_shared_folder(mdb_conn, folder_ai, user_ai):
    def apply_mdb_op(Cls, uid, **kwargs):
        return Cls(mdb_conn, uid)(**kwargs)

    init_user_if_not_exists(mdb_conn, user_ai.uid)
    owner_folder = MdbQueries(mdb_conn, folder_ai.uid).folder_by_type('inbox')

    apply_mdb_op(
        OPS.AddFolderToSharedFolders,
        folder_ai.uid,
        fid=owner_folder.fid,
    ).commit()

    create_subscribed_folder_op = OPS.CreateFolder(mdb_conn, user_ai.uid)(
        name=folder_ai.login,
        parent_fid=None
    )
    create_subscribed_folder_op.commit()
    subscriber_folder = create_subscribed_folder_op.result

    apply_mdb_op(
        OPS.AddFolderToSubscribedFolders,
        user_ai.uid,
        fid=subscriber_folder.fid,
        owner_uid=folder_ai.uid,
        owner_fid=owner_folder.fid,
    ).commit()

    apply_mdb_op(
        OPS.AddSubscriberToSharedFolders,
        folder_ai.uid,
        fid=owner_folder.fid,
        subscriber=user_ai.uid,
    )

    print('User %r subscribed to %r' % (user_ai, folder_ai))


def subscribe_user(args):
    folder_ai = get_all_by_uid(DOCKER.blackbox, args.shared_folder_uid)
    try:
        user_ai = get_all_by_uid(DOCKER.blackbox, args.subscriber_uid)
    except NonExistentUserError:
        log.info('Subscriber not exists - try init him')
        user_ai = create_user_in_docker(args.subscriber_uid, 'subscriber-%d' % args.subscriber_uid)

    with transaction(MDB_DSN) as mdb_conn:
        subscriber_user_to_shared_folder(mdb_conn, folder_ai, user_ai)


def main():
    parser = ArgumentParser(
    )
    subparsers = parser.add_subparsers()
    copy_parser = subparsers.add_parser(
        'copy',
        help='help copy user from load into docker env'
    )
    copy_parser.add_argument(
        'login',
        help='user login in load (https://paste.yandex-team.ru/118367)'
    )
    copy_parser.add_argument(
        '--uid',
        type=long,
        help='uid in docker (if not set, use same as in load)'
    )
    copy_parser.set_defaults(
        func=copy_from_load
    )

    subscribe_parser = subparsers.add_parser(
        'subscribe',
        help='subscribe user to shared folder'
    )
    subscribe_parser.add_argument(
        'shared_folder_uid',
        type=long,
    )
    subscribe_parser.add_argument(
        'subscriber_uid',
        type=long,
    )
    subscribe_parser.set_defaults(
        func=subscribe_user,
    )

    parser.add_argument(
        '-v', '--verbose',
        action='store_true',
        help='print queries and debug'
    )

    args = parser.parse_args()
    init_logging(logging.DEBUG if args.verbose else logging.WARNING)
    args.func(args)

if __name__ == '__main__':
    main()
