CREATE OR REPLACE FUNCTION util.fix_defaults(i_uid code.uid, current_revision bigint)
    RETURNS void AS $$
DECLARE
    fx_labels  mail.labels[];
    fx_folders mail.folders[] := ARRAY[]::mail.folders[];
    fld        mail.folders;
    i_serials  mail.serials;
    c          record;
BEGIN
    SELECT *
      INTO i_serials
      FROM mail.serials
     WHERE uid = i_uid;

    FOR c IN (
        SELECT df.name, df.type,
               f_same_name.fid AS same_name_fid,
               f_same_name.type AS same_name_type
          FROM code.default_folders() df
          LEFT JOIN mail.folders f_same_name
            ON (df.name = f_same_name.name
                AND f_same_name.uid=i_uid
                AND f_same_name.parent_fid IS NULL)
         WHERE NOT EXISTS (
            SELECT 1
              FROM mail.folders f
             WHERE f.type = df.type
               AND f.uid = i_uid)
    ) LOOP
        IF c.same_name_type != 'user'::mail.folder_types THEN
            -- NULL ok, cause NULL !=  -- evaluate to NULL
            RAISE EXCEPTION
                'Unexpected behavior % folder (name: %, fid: %) already has system type: %',
                    c.type, c.name, c.same_name_fid, c.same_name_type
                USING HINT = 'Show it to developers';
        END IF;
        IF c.same_name_fid IS NOT NULL THEN
            UPDATE mail.folders f
               SET type = c.type,
                   unique_type = code.is_unique_folder_type(c.type)
             WHERE f.fid = c.same_name_fid
               AND f.uid = i_uid
            RETURNING * INTO fld;
        ELSE
            INSERT INTO mail.folders
                (uid,
                fid,
                revision,
                name, type,
                unique_type,
                uidvalidity)
            VALUES
                (i_uid,
                i_serials.next_fid + cardinality(fx_folders),
                current_revision,
                c.name, c.type,
                code.is_unique_folder_type(c.type),
                code.folder_uidvalidity(current_timestamp))
            RETURNING * INTO fld;
        END IF;
        fx_folders := array_append(fx_folders, fld);
    END LOOP;

    IF cardinality(fx_folders) > 0 THEN
        INSERT INTO mail.fix_log
            (uid, revision,
            fix_key, fixed)
        VALUES
            (i_uid, current_revision,
            'folders.missed_system', to_json(fx_folders)::jsonb);

        -- actually don't need to increment next_fid,
        -- for folders with `recovered` type
        -- but it's not a problem
        UPDATE mail.serials
           SET next_fid = next_fid + cardinality(fx_folders)
         WHERE uid = i_uid;
    END IF;

    WITH labels_fixed AS (
        INSERT INTO mail.labels
            (uid, lid, revision,
            name, type)
        SELECT
            i_uid, rn + i_serials.next_lid - 1, current_revision,
            ld.name, ld.type
         FROM (
            SELECT idl.*, row_number() over() AS rn
              FROM (
                SELECT name, type
                  FROM code.default_labels() dl
                EXCEPT
                SELECT name, type
                  FROM mail.labels
                 WHERE uid = i_uid) idl) ld
        RETURNING *)
    SELECT array_agg(fl)
      INTO fx_labels
      FROM labels_fixed fl;

    IF cardinality(fx_labels) > 0 THEN
        INSERT INTO mail.fix_log
            (uid, revision, fix_key, fixed)
        VALUES
            (i_uid, current_revision, 'labels.missed_system', to_json(fx_labels)::jsonb);
        UPDATE mail.serials
           SET next_lid = next_lid + cardinality(fx_labels)
         WHERE uid = i_uid;
    END IF;

END;
$$ LANGUAGE plpgsql;
