CREATE OR REPLACE FUNCTION util.fix_folders(i_uid code.uid, current_revision bigint)
    RETURNS void AS $$
DECLARE
BEGIN
    WITH fd_up AS (
        UPDATE mail.folders fd
           SET message_count = real_message_count,
               message_seen = real_message_seen,
               message_size = real_message_size,
               message_recent = real_message_recent,
               attach_count = real_attach_count,
               attach_size = real_attach_size,
               revision = current_revision,
               next_imap_id = greatest(next_imap_id, max_imap_id+1),
               first_unseen = (real_fu).first_unseen,
               first_unseen_id = (real_fu).first_unseen_id
          FROM (
            SELECT f.fid AS fid,
                   coalesce(real_message_count, 0) AS real_message_count,
                   coalesce(real_message_seen, 0) AS real_message_seen,
                   coalesce(real_message_size, 0) AS real_message_size,
                   coalesce(real_message_recent, 0) AS real_message_recent,
                   coalesce(real_attach_count, 0) AS real_attach_count,
                   coalesce(real_attach_size, 0) AS real_attach_size,
                   coalesce(max_imap_id, 0) AS max_imap_id,
                   coalesce(max_revision, 0) AS max_revision,
                   impl.find_first_unseen(i_uid, f.fid) AS real_fu
              FROM mail.folders f
              LEFT JOIN (
                SELECT uid,
                       fid,
                       count(*) as real_message_count,
                       sum(seen::integer) as real_message_seen,
                       sum(size) as real_message_size,
                       sum(recent::integer) as real_message_recent,
                       sum(coalesce(array_length(attaches, 1), 0)) as real_attach_count,
                       sum(code.get_attaches_size(attaches)) as real_attach_size,
                       max(imap_id) as max_imap_id,
                       max(revision) as max_revision
                  FROM mail.box mb
                  JOIN mail.messages mm
                 USING (uid, mid)
                 WHERE mb.uid = i_uid
                 GROUP BY uid, fid) fr
             USING (uid, fid)
             WHERE f.uid = i_uid) r
        WHERE fd.uid = i_uid
          AND fd.fid = r.fid
          AND (
               message_count != real_message_count
            OR message_seen != real_message_seen
            OR message_recent != real_message_recent
            OR message_size != real_message_size
            OR attach_count != real_attach_count
            OR attach_size != real_attach_size
            OR next_imap_id <= max_imap_id
            OR revision < max_revision
            OR first_unseen != (real_fu).first_unseen
            OR first_unseen_id IS DISTINCT FROM (real_fu).first_unseen_id)
        RETURNING fd.fid)
    INSERT INTO mail.fix_log
        (uid, revision,
        fix_key, fixed)
    SELECT
        i_uid, current_revision,
        'folders.fixed_counters', to_json(fixed_fids)::jsonb
      FROM (
        SELECT array_agg(fid) AS fixed_fids
          FROM fd_up) x
     WHERE cardinality(fixed_fids) > 0;
END;
$$ LANGUAGE plpgsql;
