CREATE OR REPLACE FUNCTION util.fix_labels(i_uid code.uid, current_revision bigint)
    RETURNS void AS $$
DECLARE
    c              record;
    missed_lids    int[] := ARRAY[]::int[];
BEGIN
    FOR c IN (
        SELECT m_lid,
               array_agg(mid) AS mids
          FROM mail.box, unnest(lids) AS m_lid
         WHERE uid = i_uid
           AND NOT EXISTS (
            SELECT 1
              FROM mail.labels ld
             WHERE uid = i_uid
               AND ld.lid = m_lid)
         GROUP BY m_lid)
    LOOP
        UPDATE mail.box
           SET lids = lids - ARRAY[c.m_lid]
         WHERE uid = i_uid
           AND mid = ANY(c.mids);
        missed_lids := array_append(missed_lids, c.m_lid);
    END LOOP;

    IF #missed_lids > 0 THEN
        INSERT INTO mail.fix_log
            (uid, revision, fix_key, fixed)
        VALUES
            (i_uid, current_revision,
            'box.missed_lids', to_json(missed_lids)::jsonb);
    END IF;

    UPDATE mail.threads
       SET labels = ARRAY[]::mail.thread_label[]
     WHERE uid = i_uid;

    UPDATE mail.threads t
       SET labels = r_labels
      FROM (
        SELECT tid,
               impl.sum(impl.make_thread_labels(lids, 1)) r_labels
          FROM mail.box
         WHERE uid = i_uid
           AND tid IS NOT NULL
           AND #lids > 0
         GROUP BY tid) m
     WHERE t.tid = m.tid
       AND t.uid = i_uid;

    WITH ld_up AS (
        UPDATE mail.labels ld
           SET message_count = real_message_count,
               revision = current_revision,
               message_seen = CASE message_seen WHEN NULL THEN NULL ELSE real_message_seen END
          FROM (
            SELECT ld.uid,
                   ld.lid,
                   coalesce(real_message_count, 0) AS real_message_count,
                   coalesce(real_message_seen, 0) AS real_message_seen
              FROM mail.labels ld
              LEFT JOIN (
                SELECT i_uid AS uid,
                       lid,
                       count(*) AS real_message_count,
                       sum(seen::integer) AS real_message_seen
                  FROM (
                    SELECT unnest(lids) AS lid, seen AS seen
                      FROM mail.box
                     WHERE uid = i_uid
                       AND tid IS NOT NULL) m
                 GROUP BY lid) as rl
             USING (uid, lid)
             WHERE ld.uid = i_uid) r
        WHERE ld.uid = i_uid
          AND ld.lid = r.lid
          AND (message_count != real_message_count OR message_seen IS NOT NULL AND message_seen != real_message_seen)
        RETURNING ld.lid)
    INSERT INTO mail.fix_log
        (uid, revision,
        fix_key, fixed)
    SELECT
        i_uid, current_revision,
        'labels.fixed_counters', to_json(fixed_lids)::jsonb
      FROM (
        SELECT array_agg(lid) AS fixed_lids
          FROM ld_up) x
     WHERE cardinality(fixed_lids) > 0;
END;
$$ LANGUAGE plpgsql;
