CREATE OR REPLACE FUNCTION util.fix_threaded_layout(i_uid code.uid)
    RETURNS void AS $$
DECLARE
    v_problem_mids  bigint[];
    v_problem_fid   int;
    v_threaded_fid  int;
    v_unthreaed_fid int;
    v_chunk_n       int;
    c_chunk_size    constant int := 1000;
BEGIN
    IF NOT code.is_threaded_folder('inbox') OR code.is_threaded_folder('trash') OR code.is_threaded_folder('pending') THEN
        RAISE EXCEPTION 'Unexpected behavior, inbox became unthreaded or trash became threaded or pending became threaded'
             USING HINT = 'Fix that code';
    END IF;

    SELECT fid
      INTO v_threaded_fid
      FROM mail.folders
     WHERE uid = i_uid
       AND type = 'inbox';

    SELECT fid
      INTO v_unthreaed_fid
      FROM mail.folders
     WHERE uid = i_uid
       AND type = 'trash';


    IF v_threaded_fid IS NULL OR v_unthreaed_fid IS NULL THEN
        RAISE EXCEPTION 'Unexpected behavior,
                         user does not have trash %
                         or inbox % folder', v_unthreaed_fid, v_threaded_fid;
    END IF;


    -- messages in threaded folders without tids
    FOR v_problem_fid, v_chunk_n, v_problem_mids IN (
        SELECT fid, row_n / c_chunk_size as chunk_n, array_agg(mid) AS mids
          FROM (
            SELECT fid, mid, row_number() OVER () as row_n
              FROM mail.box b
              JOIN mail.folders f
             USING (uid, fid)
             WHERE f.uid = i_uid
               AND code.is_threaded_folder(f.type)
               AND b.tid IS NULL
        ) numbered
        GROUP BY fid, chunk_n
    ) LOOP
        -- move to unthreaded and back
        PERFORM util.drag_messages(
            i_uid  := i_uid,
            i_mids := v_problem_mids,
            i_source_fid := v_problem_fid,
            i_dest_fid := v_unthreaed_fid
        );
    END LOOP;

    -- messages in unthreaded folders with tids
    FOR v_problem_fid, v_chunk_n, v_problem_mids IN (
        SELECT fid, row_n / c_chunk_size as chunk_n, array_agg(mid) AS mids
          FROM (
            SELECT fid, mid, row_number() OVER () as row_n
              FROM mail.box b
              JOIN mail.folders f
             USING (uid, fid)
             WHERE f.uid = i_uid
               AND NOT code.is_threaded_folder(f.type)
               AND b.tid IS NOT NULL
        ) numbered
        GROUP BY fid, chunk_n
    ) LOOP
        PERFORM util.drag_messages(
            i_uid  := i_uid,
            i_mids := v_problem_mids,
            i_source_fid := v_problem_fid,
            i_dest_fid := v_threaded_fid
        );
    END LOOP;
END;
$$ LANGUAGE plpgsql;
