CREATE TYPE code.t_chunk AS (
    chunk_id uuid,
    task_info json,
    mids bigint [ ] );

CREATE OR REPLACE FUNCTION code.add_task (
    i_uid bigint,
    i_task_id uuid,
    i_task_info json,
    i_chunks json )
RETURNS void AS $$ DECLARE cid uuid;
chunks json [ ];
jchunk json;
mids bigint [ ];
task_size integer;
BEGIN
    INSERT INTO operations.tasks (
        task_id,
        uid,
        task_info )
    VALUES (
        i_task_id,
        i_uid,
        i_task_info );
task_size := 0;
chunks := ARRAY (
    SELECT
        JSON_ARRAY_ELEMENTS (
            i_chunks ) );
FOREACH jchunk IN ARRAY chunks LOOP cid := jchunk ->> 'id';
mids := ARRAY (
    SELECT
        JSON_ARRAY_ELEMENTS (
            jchunk -> 'mids' ) );
task_size := task_size + cardinality(mids);
INSERT INTO operations.chunks (
    task_id,
    chunk_id )
VALUES (
    i_task_id,
    cid );
INSERT INTO operations.mids (
    chunk_id,
    mid )
SELECT
    cid,
    i_mid
FROM
    UNNEST (
        mids )
    i_mid;
END LOOP;
INSERT INTO operations.recent_tasks (
    task_id,
    SIZE,
    uid )
VALUES (
    i_task_id,
    task_size,
    i_uid );
END;
$$ LANGUAGE plpgsql
SET enable_seqscan TO 'off';

CREATE OR REPLACE FUNCTION code.remove_chunk (
    i_uid bigint,
    i_chunk_id uuid,
    i_hostname text )
RETURNS void AS $$ DECLARE l_task_id uuid;
BEGIN
SELECT
    operations.chunks.task_id INTO l_task_id
FROM
    operations.chunks
WHERE
    chunk_id = i_chunk_id;
DELETE
FROM
    operations.chunks
WHERE
    chunk_id = i_chunk_id;
INSERT INTO operations.change_log (
    uid,
    TYPE,
    changed,
    hostname )
VALUES (
    i_uid,
    'delete-chunk',
    JSON_BUILD_OBJECT (
        'chunk_id',
        i_chunk_id ) ::jsonb,
    i_hostname );
DELETE
FROM
    operations.tasks task
WHERE
    NOT EXISTS (
        SELECT
            1
        FROM
            operations.chunks chunk
        WHERE
            task.task_id = chunk.task_id )
    AND task.task_id = l_task_id;
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION code.get_chunks (
    i_uid bigint,
    i_chunk_ids uuid [ ] )
RETURNS SETOF code.t_chunk AS $$
BEGIN
    RETURN query
SELECT
    chunk_id,
    task_info,
    (
        SELECT
            ARRAY_AGG (
                mid )
        FROM
            operations.mids
        WHERE
            chunk_id = c.chunk_id ) AS mids
FROM
    operations.chunks c INNER JOIN operations.tasks USING (
        task_id )
WHERE
    c.chunk_id = ANY (
        i_chunk_ids );
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION code.can_add_task (
    i_uid bigint,
    i_tasks_limit integer )
RETURNS boolean AS $$ DECLARE l_is_banned boolean;
l_count integer;
BEGIN
SELECT
    true INTO l_is_banned
FROM
    operations.banned_users
WHERE
    uid = i_uid;
IF l_is_banned THEN RETURN false;
ELSE DELETE
FROM
    operations.recent_tasks
WHERE
    uid = i_uid
    AND created < NOW ( )
    - INTERVAL '1h';
SELECT
    COALESCE (
        SUM (
            SIZE ),
        0 )
    INTO l_count
FROM
    operations.recent_tasks
WHERE
    uid = i_uid;
RETURN l_count < i_tasks_limit;
END IF;
END;
$$ LANGUAGE plpgsql;

CREATE TYPE code.user_stat_result AS (
    o_task_id uuid,
    o_task_info json,
    o_created integer,
    o_count bigint
);

CREATE OR REPLACE FUNCTION code.user_stat (
    i_uid bigint )
RETURNS SETOF code.user_stat_result AS $$
BEGIN
    RETURN query
SELECT
    task_id,
    task_info,
    CAST (
        FLOOR (
            EXTRACT (
                epoch
                FROM
                t.created ) ) AS integer ),
    COUNT (
        chunk_id )
FROM
    operations.tasks t INNER JOIN operations.chunks USING (
        task_id )
WHERE
    uid = i_uid
GROUP BY
    task_id;
END;
$$ LANGUAGE plpgsql;

CREATE TYPE code.get_task_result AS (
    o_task_info json,
    o_created integer,
    o_count bigint
);

CREATE OR REPLACE FUNCTION code.get_task (
    i_uid bigint,
    i_task_id uuid )
RETURNS SETOF code.get_task_result AS $$
BEGIN
    RETURN query
SELECT
    task_info,
    CAST (
        FLOOR (
            EXTRACT (
                epoch
                FROM
                t.created ) ) AS integer ),
    COUNT (
        chunk_id )
FROM
    operations.tasks t INNER JOIN operations.chunks USING (
        task_id )
WHERE
    task_id = i_task_id
GROUP BY
    task_id;
END;
$$ LANGUAGE plpgsql;

CREATE TYPE code.get_chunks_by_mids_result AS (
    o_chunk_id uuid,
    o_task_info json,
    o_mids bigint [ ]
);

CREATE OR REPLACE FUNCTION code.get_chunks_by_mids (
    i_uid bigint,
    i_mids bigint [ ] )
RETURNS SETOF code.get_chunks_by_mids_result AS $$
BEGIN
    RETURN query
SELECT
    chunk_id,
    task_info,
    (
        SELECT
            ARRAY_AGG (
                mid )
        FROM
            operations.mids
        WHERE
            chunk_id = c.chunk_id ) AS mids
FROM
    operations.chunks c INNER JOIN operations.tasks t USING (
        task_id )
WHERE
    uid = i_uid
    AND chunk_id IN (
        SELECT
            chunk_id
        FROM
            operations.mids
        WHERE
            mid = ANY (
                i_mids ) );
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION code.choose_users (
    i_limit integer )
RETURNS SETOF bigint AS $$
BEGIN
    RETURN query
SELECT
    uid
FROM
    operations.tasks t
WHERE
    EXISTS (SELECT 1
              FROM operations.chunks c
             WHERE c.task_id = t.task_id)
GROUP BY
    uid
ORDER BY
    MIN (
        created )
LIMIT i_limit;
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION code.has_tasks (
    i_uid bigint )
RETURNS boolean AS $$
BEGIN
    RETURN EXISTS (
        SELECT
            1
        FROM
            operations.tasks
        WHERE
            uid = i_uid );
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION code.choose_chunk_ids (
    i_uid bigint,
    i_limit integer )
RETURNS SETOF uuid AS $$
BEGIN
    RETURN query
SELECT
    chunk_id
FROM
    operations.tasks t INNER JOIN operations.chunks c USING (
        task_id )
WHERE
    uid = i_uid
LIMIT i_limit;
END;
$$ LANGUAGE plpgsql;
