/**
* Adds an operation with given unique id into the database. The operation
* will be added in a 'fresh' state. If there is another operation with the
* same unique id, the operation will not be added, and the corresponding
* returning value will be false. A user is being responsible to find the
* reason for the id conflict if necessary.
*
* @param i_id -- unique id of the operation, should be unique in whole DB,
* @param i_type -- operation type,
* @param i_uid -- user id for whose metadata the operation will be applied,
* @param i_data -- operation-related data
* @param i_request_id -- request id to be stored in change log entry
*
* @returns a composite with insertion success indicator (inserted) and the
*          actual operation is related to the specified i_id (operation).
*/
CREATE OR REPLACE FUNCTION code.add_operation (
    i_id            uuid,
    i_type          mops.operation_type,
    i_uid           bigint,
    i_data          jsonb,
    i_request_id    text)
RETURNS code.add_operation_result AS $$
DECLARE
    v_row           mops.operations;
BEGIN
    PERFORM pg_advisory_xact_lock(i_uid);

    SELECT *
      INTO v_row
      FROM mops.operations
     WHERE id = i_id;

    IF found THEN
        RETURN (false, v_row)::code.add_operation_result;
    END IF;

    INSERT INTO mops.operations (id, uid, type, data)
        VALUES (i_id, i_uid, i_type, i_data)
    RETURNING *
    INTO v_row;

    PERFORM impl.write_change_log (
        i_type          => 'operation_create'::mops.change_type,
        i_uid           => i_uid,
        i_op_id         => i_id,
        i_changed       => json_build_object(
                                'id', i_id,
                                'type', i_type,
                                'data', i_data
                            )::jsonb,
        i_request_id    => i_request_id
    );
    RETURN (true, v_row)::code.add_operation_result;
END;
$$ LANGUAGE plpgsql;
