CREATE OR REPLACE FUNCTION code.acquire_lock (
    i_uid         bigint,
    i_ttl         interval,
    i_launch_id   text,
    i_hostname    text)
RETURNS boolean AS $$
DECLARE
    v_existing_lock     operations.user_locks;
    v_acquired_lock     operations.user_locks;
BEGIN
    -- Get existing lock for user to check if it has expired or if it was acquired by the same caller
    SELECT * INTO v_existing_lock
        FROM operations.user_locks
        WHERE uid = i_uid
        FOR UPDATE;

    IF NOT found THEN
        -- There was no lock for this user, so we create one
        INSERT INTO operations.user_locks (
                uid,
                ttl,
                launch_id,
                hostname)
            VALUES (
                i_uid,
                i_ttl,
                i_launch_id,
                i_hostname)
            ON CONFLICT DO NOTHING
            RETURNING * INTO v_acquired_lock;

        IF NOT found THEN
            -- Insert was not successful due to conflict. It is possible if calls was concurrent. If so we cannot acquire lock.
            RETURN false;
        END IF;

    ELSIF (v_existing_lock.launch_id = i_launch_id) OR  -- < This lock was acquired earlier by same caller
          ((v_existing_lock.heartbeated + v_existing_lock.ttl) < now()) THEN  -- < This lock is expired
        UPDATE operations.user_locks
            SET ttl = i_ttl,
                heartbeated = now(),
                -- Update assigned only if caller is not the same
                assigned = CASE WHEN v_existing_lock.launch_id != i_launch_id THEN now()
                                ELSE v_existing_lock.assigned END,
                launch_id = i_launch_id,
                hostname = i_hostname
            WHERE uid = i_uid
            RETURNING * INTO v_acquired_lock;

    ELSE
        -- We cannot acquire lock, so return false
        RETURN false;
    END IF;

    -- Here we have successfully acquired lock, so we need to log this
    INSERT INTO operations.change_log (
            uid,
            TYPE,
            changed,
            hostname)
        VALUES (
            v_acquired_lock.uid,
            'lock-acquire',
            row_to_json(v_acquired_lock)::jsonb,
            v_acquired_lock.hostname);

    RETURN true;

END;
$$ LANGUAGE plpgsql;
