# coding: utf-8

import os
import sys
import psycopg2
import logging
from logging.handlers import WatchedFileHandler


class BaseCron(object):
    logger = logging.getLogger(__name__)

    def __init__(self, conf):
        self.conf = conf

    def run(self):
        with self._get_db_conn() as conn:
            try:
                return self._run(conn)
            except psycopg2.Warning as exc:
                self.logger.warning('PG_WARNING: %s', exc)
            except psycopg2.Error as exc:
                self.logger.error('PG_ERROR: %s', exc)
            except Exception as exc:
                self.logger.error('ERROR: %s: %s', type(exc).__name__, exc)

    def _run(self, conn):
        raise NotImplementedError('cron not implemented')

    def _get_db_conn(self):
        try:
            conn = psycopg2.connect(**(self.conf['conninfo']))
        except psycopg2.Error:
            self.logger.exception('Could not connect to DB')
            sys.exit(1)
        return conn


def init_logging(conf, log_file_name):
    log_dir = conf.get('log_dir')
    if log_dir and log_file_name:
        log_file = os.path.join(log_dir, log_file_name)
        handler = WatchedFileHandler(log_file)
    else:
        handler = logging.StreamHandler(sys.stdout)

    formatter = logging.Formatter('%(asctime)s [%(levelname)s] %(message)s')
    handler.setFormatter(formatter)

    root_logger = logging.getLogger()
    root_logger.addHandler(handler)
    root_logger.setLevel(logging.INFO)


def get_cron(name, conf):
    crons = {
        c.name: c for c in BaseCron.__subclasses__()
    }
    return crons.get(name, BaseCron)(conf)
