/**
* Performs chunk automata transitions, according to given
* operation id, chunk id and action to apply. Writes entry into change log
* in case of chunk's state has been changed.
*
* @note Unlike `impl.transit_operation_state()` this function
*       *DOES NOT* support concurrent call. It is supposed that
*       the only one worker would handle operation with all the chunks,
*       thus only this worker should change the state of the chunk.
*
* @param i_op_id -- unique id of the operation
* @param i_id -- id of the chunk
* @param i_action -- action is applying to the operation
* @param i_request_id -- request id to be stored in change log entry
*
* @return chunk state after the action application.
*
* @throws exception on inapropriate action if no transition found.
*/
CREATE OR REPLACE FUNCTION impl.transit_chunk_state(
    i_op_id         uuid,
    i_id            bigint,
    i_action        impl.chunk_action,
    i_request_id    text
) RETURNS mops.chunk_state AS $$
DECLARE
    v_from_state    mops.chunk_state;
    v_new_state     mops.chunk_state;
    v_uid           bigint;
BEGIN
    SELECT state INTO v_from_state
      FROM mops.message_chunks
     WHERE op_id = i_op_id AND id = i_id;

    SELECT to_state
      INTO v_new_state
      FROM impl.chunk_transitions()
     WHERE action = i_action
       AND from_state = v_from_state;

    IF NOT found THEN
        RAISE EXCEPTION 'no transition from % state with % action', v_from_state, i_action;
    END IF;

    IF v_from_state IS DISTINCT FROM v_new_state THEN
        SELECT uid INTO v_uid
          FROM mops.operations
         WHERE id = i_op_id;

        UPDATE mops.message_chunks
           SET state = v_new_state
         WHERE op_id = i_op_id AND id = i_id;

        PERFORM impl.write_change_log (
            i_type      => 'chunk_change_state'::mops.change_type,
            i_uid       => v_uid,
            i_op_id     => i_op_id,
            i_changed   => json_build_object(
                'id',           i_id,
                'action',       i_action,
                'from_state',   v_from_state,
                'new_state',    v_new_state
            )::jsonb,
            i_request_id => i_request_id
        );
    END IF;

    RETURN v_new_state;
END;
$$ LANGUAGE plpgsql;