/**
* Performs operation automata transitions, according to given
* operation id and action to apply. Writes entry into change log
* in case of operation's state has been changed.
*
* @note This operation supports concurrent call due to lock operation
* via `impl.acquire_operation()` function and CAS semantic
*
* @param i_id -- unique id of the operation
* @param i_action -- action is applying to the operation
* @param i_request_id -- request id to be stored in change log entry
*
* @return operation after the action application.
*
* @throws exception on inapropriate action if no transition found.
*/
CREATE OR REPLACE FUNCTION impl.transit_operation_state(
    i_id            uuid,
    i_action        impl.operation_action,
    i_request_id    text
) RETURNS mops.operations AS $$
DECLARE
    v_row           mops.operations;
    v_from_state    mops.operation_state;
    v_new_state     mops.operation_state;
    v_is_recent     boolean;
BEGIN
    v_row := impl.acquire_operation(i_id);
    v_from_state := v_row.state;

    SELECT max(seq_id) = v_row.seq_id
      INTO v_is_recent
      FROM mops.operations
     WHERE uid = v_row.uid;

    SELECT to_state
      INTO v_new_state
      FROM impl.operation_transitions()
     WHERE action = i_action
       AND from_state = v_from_state
       AND (is_recent = v_is_recent OR is_recent IS NULL);

    IF NOT found THEN
        RAISE EXCEPTION 'no transition from % state with % action', v_row.state, i_action;
    END IF;

    IF v_from_state = 'complete' AND i_action = 'delete_operation' AND (now() - v_row.created) > impl.operation_ttl() THEN
        v_new_state := 'end';
    END IF;

    IF v_from_state IS DISTINCT FROM v_new_state THEN
        UPDATE mops.operations
           SET state = v_new_state
         WHERE id = i_id
        RETURNING * INTO v_row;

        PERFORM impl.write_change_log (
            i_type      => 'operation_change_state'::mops.change_type,
            i_uid       => v_row.uid,
            i_op_id     => i_id,
            i_changed   => json_build_object(
                'action',       i_action,
                'is_recent',    v_is_recent,
                'from_state',   v_from_state,
                'new_state',    v_new_state
            )::jsonb,
            i_request_id => i_request_id
        );
    END IF;

    RETURN v_row;
END;
$$ LANGUAGE plpgsql;