CREATE SCHEMA IF NOT EXISTS mops;

CREATE TYPE mops.operation_type AS enum (
    'spam',
    'unspam',
    'purge',
    'mark',
    'label',
    'unlabel',
    'create_label',
    'update_label',
    'delete_label',
    'create_folder',
    'update_folder',
    'update_folder_symbol',
    'update_folder_position',
    'delete_folder',
    'remove',
    'complex_move',
    'change_tab'
);

CREATE TYPE mops.operation_state AS enum (
    'fresh',
    'in_progress',
    'complete',
    'in_revert',
    'reverted',
    'end'
);

CREATE SEQUENCE mops.operations_seq_id_seq START WITH 1 INCREMENT BY 1 CYCLE;

CREATE TABLE mops.operations (
    id          uuid NOT NULL,
    uid         bigint NOT NULL,
    seq_id      bigint NOT NULL DEFAULT NEXTVAL ('mops.operations_seq_id_seq'),
    type        mops.operation_type NOT NULL,
    data        jsonb NOT NULL,
    created     timestamp with time zone NOT NULL DEFAULT current_timestamp,
    state       mops.operation_state NOT NULL DEFAULT 'fresh',

    CONSTRAINT pk_operations PRIMARY KEY (id)
);

CREATE INDEX i_operation_uid_seq_id ON mops.operations (uid, state, seq_id);
CREATE INDEX i_operation_state ON mops.operations (state);

CREATE TYPE mops.chunk_state AS enum (
    'initial',
    'in_progress',
    'in_revert',
    'done'
);

CREATE TABLE mops.message_chunks (
    id          bigint NOT NULL,
    op_id       uuid NOT NULL
        REFERENCES mops.operations (id) ON DELETE CASCADE,
    mids        bigint[],
    created     timestamp with time zone NOT NULL DEFAULT current_timestamp,
    state       mops.chunk_state NOT NULL DEFAULT 'initial',

    CONSTRAINT pk_chunks PRIMARY KEY (id, op_id)
);

CREATE INDEX i_message_chunks_io_op_id_state ON mops.message_chunks (op_id, state);

CREATE SEQUENCE mops.cid_seq START WITH 1 INCREMENT BY 1 CYCLE;

CREATE TYPE mops.change_type AS enum (
    'operation_create',
    'operation_change_state',
    'chunk_create',
    'chunk_change_state'
);

CREATE TABLE mops.change_log (
    cid         bigint NOT NULL DEFAULT NEXTVAL ('mops.cid_seq'),
    change_date timestamp with time zone NOT NULL DEFAULT current_timestamp,
    uid         bigint NOT NULL,
    op_id       uuid,
    type        mops.change_type NOT NULL,
    changed     jsonb,
    request_id  text NOT NULL
) PARTITION BY RANGE (change_date);

CREATE TABLE mops.change_log_templ (
    LIKE mops.change_log INCLUDING DEFAULTS
);

ALTER TABLE mops.change_log_templ
    ADD CONSTRAINT pk_change_log_templ PRIMARY KEY (cid);

CREATE INDEX i_change_log_op_id_templ ON mops.change_log_templ (op_id);

CREATE INDEX i_change_log_uid_cid_templ ON mops.change_log_templ (uid, cid);

SELECT create_parent (
    p_parent_table := 'mops.change_log',
    p_template_table := 'mops.change_log_templ',
    p_control := 'change_date',
    p_type := 'native',
    p_interval := 'daily',
    p_premake := 5,
    p_jobmon := false
);

UPDATE part_config
   SET retention = '7 days',
       retention_keep_table = false,
       retention_keep_index = false
 WHERE parent_table = 'mops.change_log';
