CREATE SCHEMA IF NOT EXISTS queue;

CREATE TYPE queue.task_state AS ENUM (
    'pending',
    'in_progress',
    'complete',
    'error'
);

CREATE TABLE queue.tasks (
    task_id             bigserial PRIMARY KEY,
    uid                 bigint NOT NULL,

    service             text NOT NULL DEFAULT current_user,
    task                text NOT NULL,
    state               queue.task_state NOT NULL DEFAULT 'pending',
    task_args           jsonb,
    reassignment_count  integer NOT NULL DEFAULT 0,
    tries               integer NOT NULL DEFAULT 0,
    try_notices         text[],

    created             timestamptz NOT NULL DEFAULT current_timestamp,
    processing_date     timestamptz NOT NULL DEFAULT current_timestamp,
    timeout             interval NOT NULL,

    worker              text,
    request_id          text,

    CONSTRAINT check_tries CHECK (
        tries >= 0
    ),

    CONSTRAINT check_state CHECK (
        state IN ('pending', 'in_progress')
    )
);

CREATE OR REPLACE FUNCTION queue.expiration_date(
    i_processing_date   timestamptz,
    i_timeout           interval
) RETURNS timestamp with time zone AS $$
    SELECT (i_processing_date + i_timeout);
$$ LANGUAGE SQL IMMUTABLE STRICT;

CREATE INDEX i_tasks_for_service_in_progress
    ON queue.tasks (service, queue.expiration_date(processing_date, timeout))
WHERE state = 'in_progress';

CREATE INDEX i_tasks_for_service_pending
    ON queue.tasks (service, processing_date)
WHERE state = 'pending';

COMMENT ON COLUMN queue.tasks.processing_date
    IS 'used primarily as the last update time, but for failed tasks used as the scheduled retry time';

COMMENT ON COLUMN queue.tasks.worker
    IS 'must be unique for each process(coroutine) performing tasks to avoid parallel execution of the task';


CREATE TABLE queue.processed_tasks (
    task_id             bigint,
    uid                 bigint,

    service             text,
    task                text,
    state               queue.task_state,
    task_args           jsonb,
    reassignment_count  integer,
    tries               integer,
    try_notices         text[],

    created             timestamptz,
    processing_date     timestamptz,
    timeout             interval,

    worker              text
) PARTITION BY RANGE (processing_date);

CREATE TABLE queue.processed_tasks_templ (
    LIKE queue.processed_tasks INCLUDING DEFAULTS
);

ALTER TABLE queue.processed_tasks_templ
    ADD CONSTRAINT pk_shard_tasks_info_templ PRIMARY KEY (task_id);

SELECT create_parent(
    p_parent_table := 'queue.processed_tasks',
    p_template_table := 'queue.processed_tasks_templ',
    p_control := 'processing_date',
    p_type := 'native',
    p_interval := 'daily',
    p_premake := 5,
    p_jobmon := false
);
UPDATE part_config
   SET retention = '30 days',
       retention_keep_table = false,
       retention_keep_index = false
 WHERE parent_table = 'queue.processed_tasks';
