CREATE SCHEMA IF NOT EXISTS shards;

CREATE TYPE shards.load_types AS ENUM(
    'cold',
    'warm',
    'hot',
    'custom',
    'dbaas_cold',
    'dbaas_warm',
    'dbaas_hot',
    'dbaas_cold1',
    'dbaas_cold2',
    'dbaas_cold3',
    'dbaas_cold4',
    'dbaas_cold5',
    'dbaas_cold6',
    'dbaas_cold7',
    'dbaas_cold8',
    'dbaas_cold9',
    'dbaas_cold10',
    'dbaas_hot1',
    'dbaas_hot2',
    'dbaas_hot3',
    'dbaas_hot4',
    'dbaas_hot5',
    'dbaas_hot6',
    'dbaas_hot7',
    'dbaas_hot8',
    'dbaas_hot9',
    'dbaas_hot10'
);

CREATE TABLE shards.shards
(
    shard_id  integer NOT NULL,
    name      text NOT NULL,
    load_type shards.load_types DEFAULT 'custom'::shards.load_types NOT NULL,
    reg_weight integer NOT NULL DEFAULT 0,

    CONSTRAINT pk_shards PRIMARY KEY (shard_id)
);

CREATE TABLE shards.users
(
    uid      bigint NOT NULL,
    shard_id integer NOT NULL
        REFERENCES shards.shards (shard_id) ON DELETE RESTRICT,
    data     jsonb,

    CONSTRAINT pk_uids PRIMARY KEY (uid),
    CONSTRAINT limit_shards_users_data CHECK (length(data::text) <= 64)
);

CREATE INDEX i_users_by_shard_id ON shards.users (shard_id);

CREATE TABLE shards.instances
(
    instance_id integer NOT NULL,
    shard_id    integer NOT NULL
        REFERENCES shards.shards (shard_id) ON DELETE CASCADE,
    host        text NOT NULL,
    port        integer NOT NULL,
    dc          text NOT NULL,
    dbname      text NOT NULL,
    CONSTRAINT pk_instances PRIMARY KEY (instance_id)
);

CREATE INDEX i_instance_shard_id ON shards.instances (shard_id);

CREATE SEQUENCE shards.shard_s;
CREATE SEQUENCE shards.instance_s;

CREATE TABLE shards.master_host
(
    master_host text NOT NULL,
    ts          timestamp with time zone DEFAULT current_timestamp,
    CONSTRAINT pk_master_host PRIMARY KEY (master_host)
);


CREATE TABLE shards.scopes (
    scope_id integer NOT NULL PRIMARY KEY,
    name     text NOT NULL CHECK (name <> '')
);


CREATE TABLE shards.scopes_by_shards (
    id         serial PRIMARY KEY,
    reg_weight integer NOT NULL DEFAULT 0,
    scope_id   integer NOT NULL
        REFERENCES shards.scopes (scope_id) ON DELETE RESTRICT,
    shard_id   integer NOT NULL
        REFERENCES shards.shards (shard_id) ON DELETE RESTRICT,

    UNIQUE (scope_id, shard_id)
);

CREATE INDEX i_scopes_by_shards_shard_id ON shards.scopes_by_shards (shard_id);


CREATE TABLE shards.shards_load
(
    shard_id        integer NOT NULL,
    load_average    integer not null default 0,
    can_transfer_to boolean not null default false,

    CONSTRAINT pk_shards_load PRIMARY KEY (shard_id),
    CONSTRAINT fk_shards_shard_id FOREIGN KEY (shard_id)
        REFERENCES shards.shards ON DELETE CASCADE
);


CREATE TABLE shards.text_users (
    uid      text NOT NULL,
    shard_id integer NOT NULL
        REFERENCES shards.shards (shard_id) ON DELETE RESTRICT,
    data     jsonb,

    CONSTRAINT pk_text_uids PRIMARY KEY (uid),
    CONSTRAINT limit_shards_text_users_data CHECK (length(data::text) <= 64)
);


CREATE TABLE shards.deleted_users (
    uid      bigint NOT NULL,
    shard_id integer NOT NULL
        REFERENCES shards.shards (shard_id) ON DELETE RESTRICT,
    data     jsonb,

    CONSTRAINT pk_deleted_users_uid PRIMARY KEY (uid),
    CONSTRAINT limit_shards_deleted_users_data CHECK (length(data::text) <= 64)
);

CREATE INDEX i_deleted_users_by_shard_id ON shards.deleted_users (shard_id);


CREATE FUNCTION shards.check_uid_for_deleted_users_is_not_in_users()
RETURNS TRIGGER LANGUAGE plpgsql AS $$
BEGIN
    IF (SELECT count(1) FROM shards.users WHERE uid=NEW.uid) > 0 THEN
        RAISE EXCEPTION 'uid % is in shards.users', NEW.uid;
    END IF;
    RETURN NEW;
END;
$$;

CREATE TRIGGER tg_deleted_users_not_in_users
    BEFORE INSERT ON shards.deleted_users
    FOR EACH ROW EXECUTE PROCEDURE shards.check_uid_for_deleted_users_is_not_in_users();


CREATE FUNCTION shards.check_uid_for_users_is_not_in_deleted_users()
RETURNS TRIGGER LANGUAGE plpgsql AS $$
BEGIN
    IF (SELECT count(1) FROM shards.deleted_users WHERE uid=NEW.uid) > 0 THEN
        RAISE EXCEPTION 'uid % is in shards.deleted_users', NEW.uid;
    END IF;
    RETURN NEW;
END;
$$;

CREATE TRIGGER tg_users_not_in_deleted_users
    BEFORE INSERT ON shards.users
    FOR EACH ROW EXECUTE PROCEDURE shards.check_uid_for_users_is_not_in_deleted_users();

CREATE SEQUENCE shards.domain_org_id START WITH 1 INCREMENT BY 1;

CREATE TABLE shards.domains_organizations (
    domain_org_id bigint NOT NULL DEFAULT nextval('shards.domain_org_id'),
    domain_id bigint,
    org_id bigint,
    shard_id integer NOT NULL
        REFERENCES shards.shards (shard_id) ON DELETE RESTRICT,

    CONSTRAINT pk_domains_organizations_domain_id PRIMARY KEY (domain_org_id),
    CONSTRAINT uk_domains_organizations_domain_id UNIQUE (domain_id),
    CONSTRAINT uk_domains_organizations_org_id UNIQUE (org_id)
);

CREATE UNIQUE INDEX uk_domain_id_for_domain
    ON shards.domains_organizations (domain_id)
 WHERE domain_id IS NOT NULL;

CREATE UNIQUE INDEX uk_org_id_for_organization
    ON shards.domains_organizations (org_id)
 WHERE org_id IS NOT NULL;
