CREATE SCHEMA IF NOT EXISTS spaniel;

CREATE TYPE spaniel.task_state AS ENUM (
    'in_progress',
    'partially_complete',
    'archived',
    'error',
    'complete'
);

CREATE TYPE spaniel.organization_state AS ENUM (
    'active',
    'disabled',
    'frozen'
);

CREATE TABLE spaniel.organizations (
    org_id       bigint                     NOT NULL PRIMARY KEY,
    state        spaniel.organization_state NOT NULL,
    doom_date    timestamptz,
    created      timestamptz                NOT NULL DEFAULT current_timestamp,

    CONSTRAINT doom_date_not_on_active CHECK (
           (state = 'active'   AND doom_date IS NULL)
        OR (state != 'active'  AND doom_date IS NOT NULL)
    )
);

CREATE TABLE spaniel.tasks (
    org_id        bigint   NOT NULL PRIMARY KEY,
    task_id       bigint   NOT NULL,

    CONSTRAINT fk_tasks_organizations FOREIGN KEY (org_id) REFERENCES spaniel.organizations ON DELETE CASCADE
);

CREATE TABLE spaniel.organization_uids (
    org_id               bigint   NOT NULL PRIMARY KEY,
    uid                  bigint[] NOT NULL,

    CONSTRAINT fk_organization_uids_organizations FOREIGN KEY (org_id) REFERENCES spaniel.organizations ON DELETE CASCADE
);

CREATE TABLE spaniel.serials (
    org_id               bigint   NOT NULL PRIMARY KEY,
    next_search_id       bigint   NOT NULL,

    CONSTRAINT fk_serials_organizations FOREIGN KEY (org_id) REFERENCES spaniel.organizations ON DELETE CASCADE
);

CREATE TABLE spaniel.searches (
    org_id              bigint    NOT NULL,
    search_id           bigint    NOT NULL,
    admin_uid           bigint    NOT NULL,

    requested_uids      bigint[]  NOT NULL,
    query               jsonb     NOT NULL,

    state               spaniel.task_state NOT NULL DEFAULT 'in_progress',
    name                text,
    notice              text,
    date_from           timestamptz NOT NULL,
    date_to             timestamptz NOT NULL,
    created             timestamptz NOT NULL DEFAULT current_timestamp,
    updated             timestamptz NOT NULL DEFAULT current_timestamp,
    request_id          text NOT NULL,

    min_search_date     timestamptz,

    CONSTRAINT pk_search PRIMARY KEY (org_id, search_id),
    CONSTRAINT fk_search_organizations FOREIGN KEY (org_id) REFERENCES spaniel.organizations ON DELETE CASCADE
);

CREATE TABLE spaniel.search_results (
    org_id        bigint      NOT NULL,
    search_id     bigint      NOT NULL,
    uid           bigint      NOT NULL,
    id            bigint      NOT NULL,
    received_date timestamptz NOT NULL,

    CONSTRAINT pk_search_results PRIMARY KEY (org_id, search_id, uid, id),
    CONSTRAINT fk_search_results_search FOREIGN KEY (org_id, search_id) REFERENCES spaniel.searches ON DELETE CASCADE
);


CREATE TYPE spaniel.action_history_type AS ENUM (
    'create_search',
    'send_share'
);

CREATE SEQUENCE spaniel.action_history_seq START WITH 1 INCREMENT BY 1 CYCLE;

CREATE TABLE spaniel.action_history (
    org_id      bigint                      NOT NULL,
    action_id   bigint                      NOT NULL DEFAULT nextval('spaniel.action_history_seq'),
    admin_uid   bigint                      NOT NULL,
    date        timestamptz                 NOT NULL DEFAULT current_timestamp,
    type        spaniel.action_history_type NOT NULL,
    info        jsonb,

    CONSTRAINT pk_action_history PRIMARY KEY (org_id, action_id),
    CONSTRAINT fk_action_history_organizations FOREIGN KEY (org_id) REFERENCES spaniel.organizations ON DELETE CASCADE
);
