#pragma once

#include <map>
#include <boost/fusion/include/std_pair.hpp>
#include <internal/query_conf/error_handler.h>
#include <internal/query_conf/spirit.h>
#include <internal/query_conf/attr_cast.h>
#include <internal/query_conf/entry.h>

namespace pgg {
namespace query {
namespace conf {

template<typename Iterator>
struct ConfSkipper: qi::grammar<Iterator> {
    ConfSkipper() : ConfSkipper::base_type(skip, "ConfSkipper") {
        skip = ascii::space
                | (ascii::char_("#") >> *(qi::char_ - qi::eol) >> qi::eol);
    }
    qi::rule<Iterator> skip;
};

struct EndpointType : qi::symbols<char, Traits::EndpointType> {
    EndpointType() {
        add
        ("auto", Traits::EndpointType::automatic)
        ("master", Traits::EndpointType::master)
        ("replica", Traits::EndpointType::replica);
    }
};

template< typename Iterator, typename Skipper >
struct Options : public qi::rule<Iterator, Entries::Query::Options(), Skipper> {
    typedef qi::rule<Iterator, Entries::Query::Options(), Skipper> Base;

    struct DebugHandler {
        typedef void result_type;
        void operator()(Traits::Options & o, bool v) const {
            o.debug = v;
        }
    };

    struct RollbackHandler {
        typedef void result_type;
        void operator()(Traits::Options & o, bool v) const {
            o.rollback = v;
        }
    };

    struct TimeoutHandler {
        typedef void result_type;
        void operator()(Traits::Options & o, const Traits::Milliseconds & v) const {
            o.timeout = v;
        }
    };

    struct EndpointHandler {
        typedef void result_type;
        void operator()(Traits::Options & o, const Traits::EndpointType & v) const {
            o.endpoint = v;
        }
    };

    Options() {
        using qi::lit;
        using qi::bool_;
        using qi::ulong_;
        using spirit::omit;
        using ascii::no_case;
        using karma::attr_cast;
        using spirit::_val;
        using spirit::_1;

        debug.name("debug");
        debug %= no_case[lit("debug")] > ':' > bool_;

        rollback.name("rollback");
        rollback %= no_case[lit("rollback")] > ':' > bool_;

        timeout.name("timeout");
        timeout %= no_case[lit("timeout")] > ':' > attr_cast(ulong_);

        endpoint.name("endpoint");
        endpoint %= no_case[lit("endpoint")] > ':' > endpointType;

        this->name("options");
        static_cast<Base&>(*this) = '[' > -((
                            debug[debugHandler(_val, _1)] |
                            rollback[rollbackHandler(_val, _1)] |
                            timeout[timeoutHandler(_val, _1)] |
                            endpoint[endpointHandler(_val, _1)]
                    ) % ',') > ']';
    }
    qi::rule<Iterator, bool(), Skipper> debug;
    qi::rule<Iterator, bool(), Skipper> rollback;
    qi::rule<Iterator, Traits::EndpointType(), Skipper> endpoint;
    qi::rule<Iterator, Traits::Milliseconds(), Skipper> timeout;
    EndpointType endpointType;
    phoenix::function< DebugHandler > const debugHandler;
    phoenix::function< RollbackHandler > const rollbackHandler;
    phoenix::function< TimeoutHandler > const timeoutHandler;
    phoenix::function< EndpointHandler > const endpointHandler;
};

template<typename Iterator, typename Skipper >
struct Body: qi::rule<Iterator, Entries::Query::Body(), Skipper> {
    typedef qi::rule<Iterator, Entries::Query::Body(), Skipper> Base;

    template <typename Part>
    struct Handler {
        typedef void result_type;
        void operator()(Traits::Body & body, const std::string & item) const {
            body.add<Part>(item);
        }
    };

    Body() {
        using qi::lit;
        using qi::lexeme;
        using ascii::char_;
        using spirit::_val;
        using spirit::_1;
        using ascii::no_case;

        var.name("variable");
        var %= lexeme[lit('$') >> +char_("a-zA-Z_0-9")];

        param.name("parameter");
        param %= lexeme[lit('$') >> lit('(') >> +char_("a-zA-Z_0-9") >> lit(')')];

        text.name("text");
        text %= lexeme[ +(char_ - char_("\"$"))];

        this->name("body");
        static_cast<Base&>(*this) = no_case[lit("query")] > lexeme[lit('"') >> *(
                    var[varHandler(_val, _1)]|
                    text[textHandler(_val, _1)]|
                    param[paramHandler(_val, _1)]) >> lit('"')];
    }
    qi::rule<Iterator, std::string()> var, param, text;

    phoenix::function< Handler<Traits::Body::Variable> > const varHandler;
    phoenix::function< Handler<Traits::Body::Parameter> > const paramHandler;
    phoenix::function< Handler<Traits::Body::Text> > const textHandler;
};

template<typename Iterator, typename Skipper >
struct ParameterMap : qi::rule<Iterator, Entries::ParameterMap(), Skipper> {
    typedef qi::rule<Iterator, Entries::ParameterMap(), Skipper> Base;

    ParameterMap() {
        using qi::lit;
        using qi::lexeme;
        using ascii::char_;

        key.name("key");
        key %= lexeme[+char_("a-zA-Z_0-9")] > ':';

        value.name("value");
        value %= lexeme[lit('"') >> *( char_ - char_('"')) >> lit('"')];

        this->name("parameter map");
        static_cast<Base&>(*this) %= lit('(') > (( key > value ) % ',') > lit(')');
    }
    qi::rule<Iterator, std::string(), Skipper> key, value;
};

template<typename Iterator, typename Skipper >
struct Parameters : qi::rule<Iterator, Entries::Parameters(), Skipper> {
    typedef qi::rule<Iterator, Entries::Parameters(), Skipper> Base;

    Parameters() {
        using qi::lit;
        using qi::lexeme;
        using ascii::char_;
        using ascii::no_case;

        parameterName.name("parameter name");
        parameterName %= lexeme[+char_("a-zA-Z_0-9")] > '=';

        this->name("parameters");
        static_cast<Base&>(*this) %= no_case[lit("parameters")] >
                    (( parameterName > parameterMap ) % ',' );
    }
    qi::rule<Iterator, std::string(), Skipper> parameterName;
    ParameterMap<Iterator, Skipper> parameterMap;
};

template<typename Iterator, typename Skipper = ConfSkipper<Iterator>,
        typename ErrorHandler = ErrorHandler<Iterator> >
struct Parser: qi::grammar<Iterator, Entries(), Skipper> {
    Parser() : Parser::base_type(start) {
        using qi::lit;
        using qi::lexeme;
        using ascii::char_;
        using qi::on_error;
        using qi::fail;
        using qi::eoi;
        using boost::spirit::ascii::no_case;

        name.name("name");
        name %= no_case[lit("sql")] > lexeme[+char_("a-zA-Z_0-9")] > ';';

        query.name("query");
        query %= -(options > ';')
                > body > ';';

        queries.name("queries");
        queries %= *(name > query);

        start.name("start");
        start %= -(parameters > ';')
                > queries > eoi;

        on_error<fail>(start,
                error(spirit::_1, spirit::_2, spirit::_3, spirit::_4));
    }
    qi::rule<Iterator, std::string(), Skipper> name;
    Body<Iterator, Skipper> body;
    qi::rule<Iterator, Entries::Query(), Skipper> query;
    qi::rule<Iterator, Entries::Queries(), Skipper> queries;
    qi::rule<Iterator, Entries(), Skipper> start;
    Options<Iterator, Skipper> options;
    Parameters<Iterator, Skipper> parameters;

    phoenix::function<ErrorHandler> const error;
};

template <typename Iter>
void parseConfig(Iter first, Iter last, Entries & out) {
    ConfSkipper<Iter> skipper;
    Parser<Iter> parser;

    Entries map;
    const bool r = phrase_parse(first, last, parser, skipper, map);
    if (!r || first != last) {
        throw std::runtime_error("parsing failed at -> " + std::string(first, last));
    }
    std::swap(map, out);
}

inline void parseConfig(const std::string &text, Entries & out) {
    parseConfig(text.begin(), text.end(), out);
}

} // namespace conf
} // namespace query
} // namespace pgg
