#pragma once

#include <boost/optional.hpp>
#include <boost/fusion/adapted/struct/define_struct.hpp>
#include <boost/fusion/sequence/intrinsic/at.hpp>

namespace pgg {

template <class Value, class Visitor, class N>
struct Field {
    static void visit(Value& value, Visitor& visitor) {
        visitor.field(boost::fusion::at<N>(value),
                      boost::fusion::extension::struct_member_name<Value, N::value>::call());
    }
};

template <class Value, class Visitor, class N>
struct Item {
    using Next = typename boost::mpl::next<N>::type;
    static void visit(Value& value, Visitor& visitor) {
        Field<Value, Visitor, N>::visit(value, visitor);
        Item<Value, Visitor, Next>::visit(value, visitor);
    }
};

template <class Value, class Visitor>
struct Item<Value, Visitor, typename boost::fusion::result_of::size<Value>::type> {
    static void visit (Value&, Visitor&) {}
};

template <class Value, class Visitor>
struct Struct : Item<Value, Visitor, boost::mpl::int_<0>> {};

template <class Row>
class RowVisitor {
public:
    RowVisitor(const Row& row) : row(row) {}

    template <class Value>
    void field(Value& value, const std::string& name) {
        row.at(name, value);
    }

    template <class Value>
    void field(boost::optional<Value>& value, const std::string& name) {
        if (row.has_column(name)) {
            row.at(name, value);
        }
    }

    template <class Value>
    void apply(Value& value) {
        Struct<Value, RowVisitor>::visit(value, *this);
    }

private:
    const Row& row;
};

template <class Row>
RowVisitor<Row> makeRowVisitor(const Row& row) {
    return RowVisitor<Row>(row);
}

template <class T, class Row>
struct CastImpl {
    static T apply(const Row& row) {
        T result;
        makeRowVisitor(row).apply(result);
        return result;
    }
};

template <class T, class Row>
T cast(const Row& row) {
    return CastImpl<T, Row>::apply(row);
}

template <typename T>
struct Cast {
    template <typename V>
    T operator()(const V& v) const { return cast<T>(v); }
};

} // namespace pgg
