#pragma once

#include <pgg/query/query.h>
#include <apq/connection_pool.hpp>
#include <pgg/error.h>
#include <pgg/range.h>
#include <pgg/profiling.h>
#include <pgg/logging.h>
#include <boost/optional.hpp>

namespace pgg {

enum class LogError {
    enable,
    disable
};

namespace database {

class error_code : public pgg::error_code {
public:
    struct info_type {
        std::string connstr;
        std::string query_name;
        std::string query_text;
        std::vector<std::string> query_values;
    };

    error_code() = default;
    explicit error_code(pgg::error_code base) : pgg::error_code(base) {}
    error_code(pgg::error_code base, boost::optional<info_type> info )
    : pgg::error_code(base), info_(info) {}

    const boost::optional<info_type> & info() const noexcept { return info_;}
    const pgg::error_code & base() const noexcept { return *this;}
private:
    boost::optional<info_type> info_;
};

//Loggers and log policies
struct LoggingAttributes {
    profiling::LogPtr profiler;
    logging::LogPtr logger;
    LogError logQueryErrorPolicy;
};

struct EndpointQuery {
    using EndpointType = query::Query::EndpointType;
    EndpointType type = EndpointType::master;
    bool forceMaster = false;
    std::size_t number = 0;
    EndpointQuery() = default;
    EndpointQuery(EndpointType type, bool forceMaster, std::size_t number)
    : type(type), forceMaster(forceMaster), number(number) {}
};

} // namespace databse


struct Connection {
    typedef std::function<void (database::error_code, apq::cursor)> RequestHandler;
    typedef std::function<void (database::error_code, DataRange)> FetchHandler;
    typedef std::function<void (database::error_code, int)> UpdateHandler;
    typedef std::function<void (database::error_code)> ExecuteHandler;
    virtual ~Connection(){}
    /**
     * Method is being used to request a query and calls handler for every row.
     * Any way, it calls the handler with empty result at the end of the result
     * data.
     */
    virtual void request(const query::Query & q, RequestHandler handler) const = 0;
    /**
     * Method is being used to request a query and calls the handler only one time
     * for all rows either the result is empty.
     */
    virtual void fetch(const query::Query & q, FetchHandler handler) const = 0;
    /**
     * Method is being used to update a data and calls the handler only one time
     * with a result.
     */
    virtual void update(const query::Query & q, UpdateHandler handler) const = 0;
    /**
     * Method is being used to execute a db procedure and calls the handler only
     * one time with a result.
     */
    virtual void execute(const query::Query & q, ExecuteHandler handler) const = 0;
};

typedef boost::shared_ptr<Connection> ConnectionPtr;
using ConnectionHandler = boost::function<void(database::error_code, ConnectionPtr)>;

struct Database : public Connection {
    /**
     * Allows to execute queries within a single connection. Connection being
     * held while the pointer exists.
     */
    virtual void withinConnection(ConnectionHandler) const = 0;
};

typedef boost::shared_ptr<Database> DatabasePtr;

} // namespace pgg
