#pragma once

#include <pgg/service/shard_resolver.h>
#include <pgg/database/pool_factory.h>
#include <pgg/database/database.h>
#include <pgg/database/endpoint.h>
#include <pgg/database/pool.h>
#include <pgg/query/query.h>

namespace pgg {
namespace database {

class ClosedOnEndpointProvider {
    EndpointPtr ptr;
    apq::connection_pool_holder conn;
public:
    ClosedOnEndpointProvider(EndpointPtr ptr) : ptr(ptr), conn(ptr->pool()) { }

    template <typename Handler>
    void provide(Handler&& onEndpoint) const {
        onEndpoint(ptr, apq::connection_pool_holder(conn));
    }
};

template <typename R>
struct Traits {
    using Resolver = R;
    using Id = typename Resolver::Id;
    using ResolverPtr = std::shared_ptr<const Resolver>;
    Id id;
    ConnectionPoolPtr connPool;
    ResolverPtr resolver;
    Traits(const Id& id, ConnectionPoolPtr cp, ResolverPtr resolver)
    : id(id), connPool(cp), resolver(resolver) {}
};

template<typename R>
struct ResolverProvider {
    using EndpointType = query::Query::EndpointType;
    ResolverProvider(Traits<R> traits, EndpointQuery endpoint)
        : traits(std::move(traits)),
          endpointType(endpoint.type),
          force(endpoint.forceMaster),
          number(endpoint.number)
    { }

    template <typename Handler>
    void provide(Handler onEndpoint) const {
        const auto pool = traits.connPool;
        const auto number = this->number;
        auto h = [onEndpoint = std::move(onEndpoint), pool, number]
                  (pgg::error_code code, std::vector<std::string> connStrs) mutable {
            if (code) {
                onEndpoint(code);
            } else if (number < connStrs.size()) {
                auto endpoint = pool->get(connStrs[number]);
                onEndpoint(endpoint);
            } else {
                std::ostringstream stream;
                stream << "endpoint number=" << number << " is too big, available only "
                       << connStrs.size() << " endpoints";
                onEndpoint(pgg::error_code(error::endpointNumberOverflow, stream.str()));
            }
        };

        auto params = typename Traits<R>::Resolver::Params(traits.id);
        traits.resolver->asyncGetConnInfo(
                params.endpointType(endpointType).force(force), std::move(h));
    }
private:
    const Traits<R> traits;
    const EndpointType endpointType;
    const bool force = false;
    const std::size_t number = 0;
};

} // namespace database
} // namespace pgg
