#pragma once

#include <pgg/database/performer/handler.h>
#include <pgg/database/performer/error_handler_invoker.h>
#include <pgg/database/endpoint_provider.h>
#include <pgg/type_name.h>
#include <pgg/range.h>

namespace pgg {
namespace database {

using Timeout = apq::time_traits::duration_type;

inline Timeout timeout(const query::Query & q, EndpointPtr endpoint) {
    static const Milliseconds noTimeout(0);
    auto t = q.timeout();
    if( t == noTimeout ) {
        t = endpoint->queryTimeout();
    }
    if( t == noTimeout ) {
        return apq::time_traits::pos_infin_duration();
    }
    return t;
}


template <typename Base, typename EndpointProvider>
class Performer : public Base, boost::noncopyable {
    using Query = query::Query;

    struct RequestMethod {
        template <typename T, typename Handler>
        static void call(T&& pool, const apq::query& q, Handler h, Timeout tm) {
            return pool.async_request_single(apq::fake_task_context(), q,
                    apq::connection_pool::request_single_handler_t(std::move(h)),
                    apq::result_format_binary, tm);
        }
    };

    struct FetchMethod {
        template <typename T, typename Handler>
        static void call(T&& pool, const apq::query& q, Handler h, Timeout tm) {
            return pool.async_request(apq::fake_task_context(), q,
                    std::move(h), apq::result_format_binary, tm);
        }
    };

    struct UpdateMethod {
        template <typename T, typename Handler>
        static void call(T&& pool, const apq::query& q, Handler h, Timeout tm) {
            return pool.async_update(apq::fake_task_context(), q,
                    std::move(h), tm);
        }
    };

    struct ExecuteMethod {
        template <typename T, typename Handler>
        static void call(T&& pool, const apq::query& q, Handler h, Timeout tm) {
            return pool.async_execute(apq::fake_task_context(), q,
                    std::move(h), tm);
        }
    };

    template <typename Method, typename Handler>
    struct Invoker {
        Handler handler;
        LoggingAttributes logAttrs;
        Query::QueryPtr q;

        void operator()(pgg::error_code err) const {
            forwardError(handler, error_code(std::move(err)));
        }

        template <typename Conn>
        void operator()(EndpointPtr endpoint, Conn&& conn) const {
            auto log = logAttrs.logger;
            auto h = makeHandler(handler, *q, std::move(logAttrs), endpoint);
            if (q->debug() && log) {
                using namespace logging;
                logDebug(*log, [&] {
                    Attributes attributes = {
                        QueryText(q->text()),
                        QueryValues(extractValues(h.query())),
                        OldMessage(toString(h.query()))
                    };

                    return Record{ q->name(), "", std::move(attributes) };
                });
            }
            Method::call(std::forward<Conn>(conn), h.query(), h, timeout(*q, endpoint));
        }

        void operator()(EndpointPtr endpoint) const {
            (*this)(endpoint, endpoint->pool());
        }
    };

    template<typename Method, typename Handler>
    void perform(const Query & query, Handler handler) const {
        generateEndpoint(Invoker<Method, Handler>{std::move(handler), loggingAttrs(), query.clone()});
    }

    EndpointProvider endpointProvider_;
    LoggingAttributes logAttrs;
protected:
    template <typename Handler>
    void generateEndpoint(Handler&& h) const {
        endpointProvider_.provide(std::forward<Handler>(h));
    }

    const LoggingAttributes& loggingAttrs() const { return logAttrs; }
public:
    Performer(EndpointProvider provider, LoggingAttributes logAttrs)
        : endpointProvider_(std::move(provider)), logAttrs(std::move(logAttrs))
    { }

    void request(const Query & query, typename Base::RequestHandler handler) const {
        perform<RequestMethod>(query, std::move(handler));
    }

    void fetch(const Query & query, typename Base::FetchHandler handler) const {
        perform<FetchMethod>(query, [handler=std::move(handler)]
                                     (error_code e, apq::row_iterator i) {
            handler(std::move(e), pgg::range(i));
        });
    }

    void update(const Query & query, typename Base::UpdateHandler handler) const {
        perform<UpdateMethod>(query, std::move(handler));
    }

    void execute(const Query & query, typename Base::ExecuteHandler handler) const {
        perform<ExecuteMethod>(query, std::move(handler));
    }
};

template <typename Provider>
using DatabaseImplBase = Performer<pgg::Database, Provider>;

template <typename Provider>
class DatabaseImpl : public DatabaseImplBase<Provider> {
    using Base = DatabaseImplBase<Provider>;
    using ConnectionImpl = Performer<pgg::Connection, ClosedOnEndpointProvider>;

    struct Invoker {
        ConnectionHandler onConnection;
        LoggingAttributes logAttrs;
        void operator()(pgg::error_code err) const {
            onConnection(error_code(std::move(err)), ConnectionPtr());
        }
        void operator()(EndpointPtr endpoint) const {
            auto conn = boost::make_shared<ConnectionImpl>(
                            ClosedOnEndpointProvider(endpoint), logAttrs);

            onConnection(error_code(), conn);
        }
    };
public:
    DatabaseImpl(Provider endpoint, LoggingAttributes logAttrs)
    : Base(std::move(endpoint), std::move(logAttrs)) {}

    void withinConnection(ConnectionHandler onConnection) const {
        this->generateEndpoint(Invoker{std::move(onConnection), this->loggingAttrs()});
    }
};

} // namespace database
} // namespace pgg
